package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toSet;

/**
 * <strong>Data Center Migration Importer SPI</strong>
 * <p>
 * Implement this interface to import data that has been exported by an {@link Exporter}.
 * <p>
 * Migration archives are read and written sequentially. Whenever an entry that was written by a corresponding
 * {@link Exporter} is encountered the correct callback on this interface is invoked.
 * <p>
 * There are two different types of entries:
 * <ul>
 *     <li>
 *         {@link #onEntry(ImportContext, EntrySource) File entries}
 *         <p>
 *         Used to migrate data that has been generated in memory, for example, JSON data.
 *     </li>
 *     <li>
 *         {@link #onArchiveEntry(ImportContext, ArchiveSource) Archive entries}
 *         <p>
 *         Used to migrate data that already exists on the file system.
 *     </li>
 * </ul>
 * {@link Importer Importers} are required to be <em>stateless</em>, as they are essentially singletons. State, which
 * should persist over the whole lifetime of a job, can be stored in {@link ImportContext#getAttributeMap()}. This data
 * is kept in memory and care should be taken that memory usage in this attribute map does not grow considerably over
 * time.
 * <strong>References to entities during the import</strong>
 * References to entities, such as {@link Repository repositories} or {@link Project projects}, are mapped by their
 * {@link EntityImportMapping#getExportId(Object) export ID} to their
 * {@link EntityImportMapping#getLocalId(String) local ID}
 * in {@link ImportContext#getEntityMapping(MigrationEntityType) an entity mapping}.
 * {@link StandardMigrationEntityType StandardMigrationEntityTypes} define mappings for
 * {@link StandardMigrationEntityType#PROJECT projects} and {@link StandardMigrationEntityType#REPOSITORY repositories}.
 * <br>
 * Custom entity types should implement the {@link MigrationEntityType} interface.
 * <p>
 * A common pattern to retrieve the {@link Repository} from an export ID is this snippet of code:
 * <pre><code>
 * Optional&lt;Repository&gt;; repo = context.getEntityMapping(StandardMigrationEntityType.REPOSITORY)
 *     .getLocalId("export ID")
 *     .map(repositoryService::getById)
 * </code></pre>
 * where {@code "export ID"} is the ID generated by an {@link Exporter} using the
 * {@link ExportContext#getEntityMapping(MigrationEntityType) export entity mapping}.
 * <p>
 * Export IDs should be encoded into the {@link EntrySource#getPath() path} of an entry whenever possible.

 * <strong>Plugin definition</strong>
 * A plugin has to define a {@link MigrationHandlerModuleDescriptor &lt;migration-handler&gt;} module in its {@code atlassian-plugin.xml} for the migration
 * process to include it. Migration handlers always define a pair of {@link Exporter exporters} and
 * {@link Importer importers}. Only entries added by the corresponding {@link Exporter} will be consumed by its
 * {@link Importer}.
 * <p>
 * Example module definition:
 * <pre><code>
 * &lt;migration-handler key="handler" weight="150"&gt;
 *     &lt;exporter class="org.foo.bar.PluginExporter"/&gt;
 *     &lt;importer class="org.foo.bar.PluginImporter"/&gt;
 * &lt;/migration-handler&gt;
 * </code></pre>
 * A migration handler's weight defines the order in which it is called in relation to other migration handlers. A higher
 * weight signifies a dependency on a lower weight handler. All core handlers have a weight lower than {@code 100}.
 *
 * @see Exporter
 * @see ImportContext
 * @see ImportContext#getEntityMapping(MigrationEntityType)
 * @since 5.13
 */
public interface Importer {

    /**
     * Set of archive versions that are supported by this version of the product.
     */
    Set<Integer> SUPPORTED_ARCHIVE_VERSIONS = Stream.of(1, 2)
            .collect(collectingAndThen(toSet(), Collections::unmodifiableSet));

    /**
     * A callback to indicate a repository and any of its dependent entities have been imported. This callback can be
     * used to perform any post-import processing related to the repository. It is preferable to using
     * {@link #onEnd(ImportContext)} because it is called frequently over the course of an import and allows for
     * for things such as temporary resources to be cleaned shortly after they are no longer needed (rather than leaving
     * them for the duration of the import) or for indexing or additional processing to be performed at each step rather
     * than left to the end where there may be a greater risk of import failure. This callback also provides the
     * certainty that {@link #onEnd(ImportContext)} cannot (without an {@link Importer} tracking state) that a
     * repository has been imported.
     *
     * @param context    the context for the import
     * @param repository the repository that was imported
     */
    default void finalizeRepositoryImport(@Nonnull ImportContext context, @Nonnull Repository repository) {
    }

    /**
     * Called when an archive entry is encountered within an archive.
     *
     * @param importContext the {@link ImportContext} for this import operation
     * @param archiveSource provides access to the data in this entry
     * @throws com.atlassian.bitbucket.migration.ImportException if importing the {@link ArchiveSource archive} fails
     */
    default void onArchiveEntry(@Nonnull ImportContext importContext, @Nonnull ArchiveSource archiveSource) {
        requireNonNull(importContext, "importContext");
        requireNonNull(archiveSource, "archiveSource");

        LoggerFactory.getLogger(getClass()).warn(
                "Unexpected archive entry on importer that does not support archives: {}", archiveSource.getPath());
    }

    /**
     * Called after the import has finished.
     *
     * @param importContext the {@link ImportContext} for this import operation
     */
    default void onEnd(@Nonnull ImportContext importContext) {
    }

    /**
     * @param importContext the {@link ImportContext} for this import operation
     * @param entrySource   provides access to the data in this entry
     * @throws com.atlassian.bitbucket.migration.ImportException if importing the {@link EntrySource entry} fails
     */
    default void onEntry(@Nonnull ImportContext importContext, @Nonnull EntrySource entrySource) {
        requireNonNull(importContext, "importContext");
        requireNonNull(entrySource, "entrySource");

        LoggerFactory.getLogger(getClass()).warn(
                "Unexpected entry on importer that does not support entries: {}", entrySource.getPath());
    }

    /**
     * Called after the export is started.
     *
     * @param importContext the {@link ImportContext} for this import operation
     */
    default void onStart(@Nonnull ImportContext importContext) {
    }
}
