package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.attribute.AttributeMap;
import com.atlassian.bitbucket.i18n.KeyedMessage;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

/**
 * Allows the consumption of entries contained in an import archive. An {@link Importer} that wishes to make state
 * available for the lifetime of an import will use the facilities provided by this class such as
 * {@link #getAttributeMap()} and {@link #getEntityMapping(MigrationEntityType)}. This class also provides a facility
 * for storing error and warning messages that an {@link Importer} may raise during the import process.
 *
 * @since 5.13
 */
public interface ImportContext {

    /**
     * Report a failed import
     *
     * @param message the failure message
     * @param entity  the entity for which export failed
     */
    void addError(@Nonnull KeyedMessage message, @Nullable Object entity);

    /**
     * Report a failed import
     *
     * @param message the failure message
     * @param entity  the entity for which export failed
     * @param t       an optional {@link Throwable} for the warning
     */
    void addError(@Nonnull KeyedMessage message, @Nullable Object entity, @Nullable Throwable t);

    /**
     * Report a warning during an import
     *
     * @param message the failure message
     * @param entity  the entity for which export failed
     */
    void addWarning(@Nonnull KeyedMessage message, @Nullable Object entity);

    /**
     * Report a warning during an import
     *
     * @param message the failure message
     * @param entity  the entity for which export failed
     * @param t       an optional {@link Throwable} for the warning
     */
    void addWarning(@Nonnull KeyedMessage message, @Nullable Object entity, @Nullable Throwable t);

    /**
     * Returns the {@link AttributeMap} object for this context. This allows storage of attributes associated with this
     * context for the lifetime of this context.
     *
     * @return the {@link AttributeMap} object for this context
     * @see AttributeMap
     */
    @Nonnull
    AttributeMap getAttributeMap();

    /**
     * Returns the current hierarchy ID the importer is importing.
     *
     * @return the current hierarchy ID the importer is importing or {@link Optional#empty} if none
     * @since 6.0
     */
    @Nonnull
    Optional<String> getCurrentHierarchyId();

    /**
     * Returns the {@link EntityImportMapping} of the {@link MigrationEntityType} for this context. If the mapping
     * does not exist for the type, it'll be created.
     *
     * @param entityType Entity type that this mapping should map
     * @param <T>        Type of the IDs in the mapping
     * @return The {@link EntityImportMapping} of the {@link MigrationEntityType} for this context
     */
    @Nonnull
    <T> EntityImportMapping<T> getEntityMapping(@Nonnull MigrationEntityType<T> entityType);

    /**
     * @return true if {@link #addError(KeyedMessage, Object)} has been called
     */
    boolean hasErrors();
}
