package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.io.IoConsumer;
import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.nio.file.Path;

/**
 * Data Center Migration Exporter SPI
 * <p>
 * Implement this interface to export data so that it can be imported by an {@link Importer}.
 * <p>
 * Migration archives are read and written sequentially. Entries added by this exporter will be read in the same order
 * by the corresponding {@link Importer}.
 * <p>
 * There are two different types of entries:
 * <ul>
 *     <li>
 *         {@link ExportSection#addEntry(Path, IoConsumer, boolean) File entries}
 *         <p>
 *         Used to migrate data that has been generated in memory, for example, JSON data.
 *     </li>
 *     <li>
 *         {@link ExportSection#addEntriesAsArchive(Path, IoConsumer, boolean) Archive entries}
 *         <p>
 *         Used to migrate data that already exists on the file system.
 *     </li>
 * </ul>
 * <strong>Important:</strong>
 * {@link Exporter Exporters} are required to be <em>stateless</em>, as they are essentially singletons. State, which
 * should persist over the whole lifetime of a job, can be stored in {@link ExportContext#getAttributeMap()}. This data
 * is kept in memory and care should be taken that memory usage in this attribute map does not grow considerably over
 * time.
 * <strong>References to entities during the export</strong>
 * References to entities, such as {@link Repository repositories} or {@link Project projects}, should be exported
 * using their {@link ExportContext#getEntityMapping(MigrationEntityType) export ID}.
 * <p>
 * {@link Importer Importers} can use this export ID to retrieve the local ID of the entity using the
 * {@link ImportContext#getEntityMapping(MigrationEntityType) import entity mapping}.
 * <p>
 * {@link StandardMigrationEntityType StandardMigrationEntityTypes} define mappings for
 * {@link StandardMigrationEntityType#PROJECT projects} and {@link StandardMigrationEntityType#REPOSITORY repositories}.
 * <br>
 * Custom entity types should implement the {@link MigrationEntityType} interface.

 * <strong>Plugin definition</strong>
 * A plugin has to define a {@link MigrationHandlerModuleDescriptor &lt;migration-handler&gt;} module in its {@code atlassian-plugin.xml} for the migration
 * process to include it. Migration handlers always define a pair of {@link Exporter exporters} and
 * {@link Importer importers}. Only entries added by the corresponding {@link Exporter} will be consumed by its
 * {@link Importer}.
 * <p>
 * Example module definition:
 * <pre><code>
 * &lt;migration-handler key="handler" weight="150"&gt;
 *     &lt;exporter class="org.foo.bar.PluginExporter"/&gt;
 *     &lt;importer class="org.foo.bar.PluginImporter"/&gt;
 * &lt;/migration-handler&gt;
 * </code></pre>
 * A migration handler's weight defines the order in which it is called in relation to other migration handlers. A higher
 * weight signifies a dependency on a lower weight handler. All core handlers have a weight lower than {@code 100}.
 *
 * @see Importer
 * @see ExportContext
 * @see ExportContext#getEntityMapping(MigrationEntityType)
 * @since 5.13
 */
public interface Exporter {

    /**
     * Archive version that this version of the product generates.
     */
    int ARCHIVE_VERSION = 2;

    /**
     * Called when a project is exported. Implementors should implement this method to export all relevant data that
     * is associated with the project.
     *
     * @param exportContext the {@link ExportContext context} for this export operation
     * @param project       the project that is exported
     */
    default void export(@Nonnull ExportContext exportContext, @Nonnull Project project) {
    }

    /**
     * Called when a pull request is exported. Implementors should implement this method to export all relevant data
     * that is associated with the pull request.
     *
     * @param exportContext the {@link ExportContext context} for this export operation
     * @param pullRequest   the pull request that is exported
     */
    default void export(@Nonnull ExportContext exportContext, @Nonnull PullRequest pullRequest) {
    }

    /**
     * Called when a repository is exported. Implementors should implement this method to export all relevant data that
     * is associated with the repository.
     *
     * @param exportContext the {@link ExportContext context} for this export operation
     * @param repository    the repository that is exported
     */
    default void export(@Nonnull ExportContext exportContext, @Nonnull Repository repository) {
    }

    /**
     * Called after the export is finished.
     *
     * @param exportContext the {@link ExportContext context} for this export operation
     */
    default void onEnd(@Nonnull ExportContext exportContext) {
    }

    /**
     * Called before the export is started.
     *
     * @param exportContext the {@link ExportContext context} for this export operation
     */
    default void onStart(@Nonnull ExportContext exportContext) {
    }
}
