package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.io.IoConsumer;

import javax.annotation.Nonnull;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.function.Consumer;

/**
 * Allows addition of entries to the export archive, either from disk or directly from a {@link OutputStream stream}.
 *
 * @since 5.13
 */
public interface ExportSection {

    /**
     * Returns a consumer for {@link SequentialArchive} that adds no entries to the archive.
     *
     * @return a consumer that adds no entries
     */
    static IoConsumer<SequentialArchive> emptyArchive() {
        return sequentialArchive -> {
        };
    }

    /**
     * Returns a consumer for {@link OutputStream} that does not write any data.
     *
     * @return a consumer that does not write any data
     */
    static IoConsumer<OutputStream> emptyEntry() {
        return outputStream -> {
        };
    }

    /**
     * Adds entries packaged as an archive to the export archive
     *
     * @param entryName the name to give this entry
     * @param writer    used to write the contents of the entry (as an archive file)
     * @param compress  whether to compress the contents of the entry
     * @throws ExportException when an unrecoverable exception occurs when writing to the archive
     */
    void addEntriesAsArchive(@Nonnull Path entryName, @Nonnull IoConsumer<SequentialArchive> writer, boolean compress);

    /**
     * Creates a file with the given {@code entryName} under this {@link ExportSection section} of the export archive,
     * then consumes the {@link OutputStream} it obtains from the file, using the given {@link Consumer}
     *
     * @param entryName the entryName of the file to be created. Use {@link Paths#get(String, String...)}
     *                  to use this method efficiently.
     * @param writer    {@link IoConsumer consumes} the {@link OutputStream} to add an entry to the archive
     * @param compress  whether to compress the entry when adding it to the archive
     * @throws ExportException when an unrecoverable exception occurs when writing to the archive
     */
    void addEntry(@Nonnull Path entryName, @Nonnull IoConsumer<OutputStream> writer, boolean compress);
}
