package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.attribute.AttributeMap;
import com.atlassian.bitbucket.i18n.KeyedMessage;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.function.Consumer;

/**
 * Represents the context for the current export job. An {@link Exporter} that wishes to make state available for the
 * lifetime of an export will use the facilities provided by this class such as {@link #getAttributeMap()} and
 * {@link #getEntityMapping(MigrationEntityType)}. This class also provides a facility for storing error and warning
 * messages that an {@link Exporter} may raise during the export process.
 *
 * @since 5.13
 */
public interface ExportContext extends ExportSection {

    /**
     * Ensures that the export has not been canceled. This should be called occasionally during long running operations
     * in order to react to cancellation requests quickly. Cleanup code can be handled in a try-catch block, but the
     * exception should be re-thrown without modifications.
     * <p>
     * Note: {@link Exporter#onEnd(ExportContext)} will always be called, even if this method throws an exception.
     * Additional cleanup can be done there as well.
     * <p>
     * Example:
     * <pre>
     * try {
     *     context.abortIfCanceled();
     * } catch (Exception e) {
     *     // cleanup code
     *     throw e;
     * }
     * </pre>
     *
     * @throws CanceledMigrationException if the job for this context has been canceled
     */
    void abortIfCanceled();

    /**
     * Report a failed export of the provided entity
     *
     * @param message the failure message
     * @param entity  the entity for which export failed, or {@code null} for general failures
     */
    void addError(@Nonnull KeyedMessage message, @Nullable Object entity);

    /**
     * Report an error during an export for the provided entity and with an optional {@link Throwable}
     * to log
     *
     * @param message the failure message
     * @param entity  the entity for which export failed, or {@code null} for general failures
     * @param t       an optional {@link Throwable} for the error
     */
    void addError(@Nonnull KeyedMessage message, @Nullable Object entity, @Nullable Throwable t);

    /**
     * Creates a {@link ExportSection section} with the given relative path inside the export archive. If a section
     * with this name has already been added in the export, it will not be added.
     *
     * @param path          a path relative to the export root directory specific to the caller, inside the export
     *                      archive. Use {@link Paths#get(String, String...)} to use this method
     *                      efficiently.
     * @param exportSection {@link Consumer consumer} that writes the contents of the {@link ExportSection section}
     *                      relative to the section path
     * @return true if the {@link ExportSection section} did not exist and has been created as a result of this call,
     *         false otherwise
     */
    boolean addSectionIfAbsent(@Nonnull Path path, @Nonnull Consumer<ExportSection> exportSection);

    /**
     * Report a warning during an export for the provided entity and with an optional {@link Throwable}
     * to log
     *
     * @param message the failure message
     * @param entity  the entity for which export failed, or {@code null} for general failures
     * @param t       an optional {@link Throwable} for the warning
     */
    void addWarning(@Nonnull KeyedMessage message, @Nullable Object entity, @Nullable Throwable t);

    /**
     * Report a warning during an export for the provided entity
     *
     * @param message the failure message
     * @param entity  the entity for which export failed, or {@code null} for general failures
     */
    void addWarning(@Nonnull KeyedMessage message, @Nullable Object entity);

    /**
     * Returns the {@link AttributeMap} object for this context. This allows storage of attributes associated with this
     * context for the lifetime of this context.
     *
     * @return the {@link AttributeMap} object for this context
     * @see AttributeMap
     */
    @Nonnull
    AttributeMap getAttributeMap();

    /**
     * Returns the {@link EntityExportMapping} of the {@link MigrationEntityType} for this context. If the mapping
     * does not exist for the type, it'll be created.
     *
     * @param entityType Entity type that this mapping should map, e.g. Repository or Project
     * @param <T>        Type of the IDs in the mapping
     * @return The {@link EntityExportMapping} of the {@link MigrationEntityType} for this context
     */
    @Nonnull
    <T> EntityExportMapping<T> getEntityMapping(@Nonnull MigrationEntityType<T> entityType);

    /**
     * @return true if {@link #addError(KeyedMessage, Object)} has been called
     */
    boolean hasErrors();

    /**
     * Does this {@link ExportContext export context} contain a {@link ExportSection section} with the given path
     *
     * @param path the path of the section to check for
     * @return true if this {@link ExportContext export context} contain a {@link ExportSection section} with the given
     *         path, false otherwise
     */
    boolean hasSection(@Nonnull Path path);
}
