package com.atlassian.bitbucket.migration;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * Provides the ability to map generated export IDs to locally created IDs during an import and vice versa.
 *
 * @since 5.13
 */
public interface EntityImportMapping<K> {

    /**
     * Adds a mapping between the <code>exportId</code> and <code>localId</code>.
     *
     * @param exportId The ID that was generated during the export
     * @param localId  The local ID for the entity on this instance
     * @throws IllegalArgumentException if either the <code>exportId</code> or the <code>localId</code> have already
     *                                  been mapped previously, either to the same value or a different value
     */
    void add(@Nonnull String exportId, @Nonnull K localId);

    /**
     * Remove all entries from this mapping
     */
    void clear();

    /**
     * Given a <code>localId</code>, return its corresponding <code>exportId</code>.
     *
     * @param localId The local ID for the entity on this instance
     * @return The export ID that was generated during the import, or {@link Optional#empty()}.
     */
    @Nonnull
    Optional<String> getExportId(@Nonnull K localId);

    /**
     * Given an <code>exportId</code>, return its corresponding <code>localId</code>.
     *
     * @param exportId The export ID that was generated during the import
     * @return The local ID for the entity on this instance, or {@link Optional#empty()}.
     */
    @Nonnull
    Optional<K> getLocalId(@Nonnull String exportId);
}
