package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.io.IoConsumer;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.nio.file.Path;
import java.util.function.Predicate;

/**
 * Provides access to an archive within an archive.
 *
 * @since 5.13
 */
public interface ArchiveSource {

    /**
     * Filter and extract this contents of this archive to the specified target.
     *
     * @param target path on disk where files will be extracted
     * @param filter predicate applied to each file before extraction. If the predicate returns {@code false} the file
     *               will not be extracted
     * @throws IOException if an error was encountered while performing IO
     */
    void extractToDisk(@Nonnull Path target, @Nonnull Predicate<String> filter) throws IOException;

    /**
     * Extract this contents of this archive to the specified target.
     *
     * @param target path on disk where files will be extracted
     * @throws IOException if an error was encountered while performing IO
     */
    default void extractToDisk(@Nonnull Path target) throws IOException {
        extractToDisk(target, p -> true);
    }

    /**
     * Get the path of this archive source.
     *
     * @return the path of this archive source
     */
    @Nonnull
    Path getPath();

    /**
     * Filter and read each file in the archive.
     *
     * @param reader used to read the contents of each file
     * @throws IOException if an error was encountered while performing IO
     */
    default void read(@Nonnull IoConsumer<EntrySource> reader) throws IOException {
        read(reader, p -> true);
    }

    /**
     * Filter and read each file in the archive.
     *
     * @param reader used to read the contents of each file
     * @param filter predicate applied to each file before reading. If the predicate returns {@code false} the file
     *               will not be extracted
     * @throws IOException if an error was encountered while performing IO
     */
    void read(@Nonnull IoConsumer<EntrySource> reader, @Nonnull Predicate<String> filter) throws IOException;
}
