package com.atlassian.bitbucket.idx;

import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Used to by {@link CommitIndexer}s to store state during the indexing process. A new context will be used for each
 * indexing run, but for each run the same context will be shared by <i>all enabled indexers</i>.
 */
public interface IndexingContext {

    /**
     * Retrieve an object from the context.
     *
     * @param key the key to lookup
     * @param <T> the type of object expected at the specified key
     * @return the value, or {@code null} if not present in the context
     */
    @Nullable
    <T> T get(@Nullable String key);

    /**
     * @return the repository that is being indexed
     */
    @Nonnull
    Repository getRepository();

    /**
     * Sets a key-value pair in the context. Keys used should be distinctive as the same context is shared across
     * all enabled {@link CommitIndexer indexers} for a given repository.
     *
     * @param key   the key to store the value under, which should be unique
     * @param value the value to store
     * @return the previous value stored under this key, or {@code null} if the key is new
     */
    @Nullable
    Object put(@Nullable String key, @Nullable Object value);

    /**
     * Removes a key-value pair from the context.
     *
     * @param key the key to remove
     * @return the value that was removed, or {@code null} if the key was not in the context
     */
    @Nullable
    Object remove(@Nullable String key);
}
