package com.atlassian.bitbucket.hook.repository;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Annotation to be used with {@link PostRepositoryHook} instances to signal that the hook prefers to be called
 * synchronously, which allows the hook to write to the SCM client error and output streams directly for
 * {@link StandardRepositoryHookTrigger#REPO_PUSH pushes} to the repository.
 * <p>
 * Note that annotating a {@link PostRepositoryHook} with {@code @SynchronousPreferred} does not guarantee that the
 * hook will be called synchronously. For many {@link RepositoryHookRequest#getTrigger() triggers} hooks will only be
 * invoked asynchronously.
 * <p>
 * If the hook should only be called synchronously, and should be skipped for asynchronous hook requests, set
 * {@link #asyncSupported()} to {@code false}.
 *
 * @see PostRepositoryHook
 * @since 5.0
 */
@Retention(RUNTIME)
@Target(TYPE)
@Documented
public @interface SynchronousPreferred {
    /**
     * @return {@code false} if the hook should only be called synchronously and should not be invoked at all for
     *         asynchronous hook requests. The default is {@code true}, which ensures that the hook is called
     *         synchronously when possible, but asynchronously when it's not.
     */
    boolean asyncSupported() default true;
}

