package com.atlassian.bitbucket.hook.repository;

import javax.annotation.Nonnull;

/**
 * A callback to receive more information about commits added or removed in a {@link RepositoryHookRequest}.
 * The callback will be called for each each relevant (ref, commit) pair:
 * <ul>
 *     <li>For {@link com.atlassian.bitbucket.repository.RefChangeType#ADD added} refs, only new commits are
 *         provided to {@link RepositoryHookCommitCallback#onCommitAdded(CommitAddedDetails)}. For new refs that point
 *         to existing commits, no commits are provided to the callback.
 *     </li>
 *     <li>For {@link com.atlassian.bitbucket.repository.RefChangeType#DELETE deleted} refs, only commits that are
 *         removed from the repository (orphaned) are provided
 *         {@link RepositoryHookCommitCallback#onCommitRemoved(CommitRemovedDetails)}.
 *     </li>
 *     <li>For {@link com.atlassian.bitbucket.repository.RefChangeType#UPDATE updated} refs, all commits that are
 *         added or removed to that ref are provided to the callback.
 *     </li>
 * </ul>
 * Multiple callbacks can occur for the same commit. For instance, when branch A is merged into branch B and branch
 * B is merged into branch C, a commit that was previously only on branch A will now be on branch A, B and C. In
 * this case, {@link RepositoryHookCommitCallback#onCommitAdded(CommitAddedDetails)} is called for both branch B and C.
 * <p>
 * The system will call {@link #onStart()} before the first commit details are provided to the callback to allow the
 * hook to initialize. Likewise, the system calls {@link #onEnd()} after the last commit details have been provided
 * to the callback. This can be because all commit details have been provided, the callback has returned {@code false}
 * from one of the callback methods to instruct the system that it's done processing commits, or - in the case of
 * {@link PreRepositoryHook pre-hooks} - because the request has been vetoed and no further commit processing will be
 * done.
 * <p>
 * If a {@link RepositoryHookRequest} is vetoed in
 * {@link PreRepositoryHook#preUpdate(PreRepositoryHookContext, RepositoryHookRequest)} before commit details are
 * retrieved, callbacks will not be called at all; the {@link #onStart()} and {@link #onEnd()} methods will not be
 * called. However, if the {@link #onStart()} method is called, the system will also call the {@link #onEnd()} method
 * at the end of processing.
 *
 * @since 5.0
 */
public interface RepositoryHookCommitCallback {

    /**
     * @param commitDetails the commit details
     * @return {@code true} if the callback wants to receive details about added and removed commits.
     *         Otherwise {@code false}. When this method returns {@code false}, neither this method nor
     *         {@link #onCommitRemoved(CommitRemovedDetails)} will be called with further commit details.
     */
    default boolean onCommitAdded(@Nonnull CommitAddedDetails commitDetails) {
        return true;
    }

    /**
     * @param commitDetails the commit details
     * @return {@code true} if the callback wants to receive details about added and removed commits.
     *         Otherwise {@code false}. When this method returns {@code false}, neither this method nor
     *         {@link #onCommitAdded(CommitAddedDetails)} will be called with further commit details.
     */
    default boolean onCommitRemoved(@Nonnull CommitRemovedDetails commitDetails) {
        return true;
    }

    /**
     * Callback method that is called after all commits have been offered to the callback. The callback can perform
     * any cleanup operations here. If {@link #onStart()} has been called, this method is guaranteed to be called
     * when the callback has finished commit processing.
     */
    default void onEnd() {
    }

    /**
     * Callback method that is called before the commits details are provided to
     * {@link #onCommitAdded(CommitAddedDetails)} and {@link #onCommitRemoved(CommitRemovedDetails)}
     */
    default void onStart() {
    }
}
