package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.repository.Ref;

import javax.annotation.Nonnull;

/**
 * Hook that is called after changes to 1 or more refs have been made.
 * <p>
 * Hooks are called asynchronously by default, but implementors can annotate their implementation class with
 * {@link SynchronousPreferred} to signal that the hook should be called synchronously. This gives the hook access
 * to the SCM client's output and error streams if the hook is triggered by a
 * {@link StandardRepositoryHookTrigger#REPO_PUSH push} to the repository.
 * <p>
 * If a hook is marked as {@link SynchronousPreferred}, the implementor should ensure that the hook does not take a
 * long time to execute, since the repository push will not finish until all hooks have completed.
 *
 * @see SynchronousPreferred
 * @since 5.0
 */
public interface PostRepositoryHook<T extends RepositoryHookRequest> {

    /**
     * Callback method that is called after a request to update one or more {@link Ref refs} is completed.
     *
     * @param context  provides hook settings and a way to obtain the commits added/removed
     * @param request  provides details about the refs that have been updated
     */
    void postUpdate(@Nonnull PostRepositoryHookContext context, @Nonnull T request);
}
