package com.atlassian.bitbucket.home;

import com.atlassian.bitbucket.ServiceException;

import javax.annotation.Nonnull;

/**
 * Invoked when the {@link com.atlassian.bitbucket.server.StorageService#getSharedHomeDir() shared home directory}
 * is updated. In general, it is a best practice not to move the shared home directory. However, for cases where a move
 * is necessary, all registered handlers are invoked to allow components which depend on the absolute path to the shared
 * home directory to update themselves to reflect its new location.
 * <p>
 * When a move is detected, the system locks access, including hosting operations such as push and pull, and invokes
 * all registered update handlers. <i>Handlers have no mechanism to prevent or revert the shared home directory change</i>;
 * that is configured outside of the application. However, if <i>any</i> registered handler throws an exception while
 * {@link #apply(HomeUpdate) applying the update}, it will prevent the system from being unlocked again. The exception
 * message will be displayed to unauthenticated users, so it <i>should not</i> contain sensitive information, such as
 * file paths. It <i>should</i> provide a high-level message about what went wrong, but the detailed information about
 * the failure should be written to the log files instead, to assist administrators in addressing the issue. When all
 * handlers have been invoked, if no exceptions were thrown, the system is unlocked and access is restored.
 * <p>
 * When multiple {@code HomeUpdateHandler}s are registered, it is possible one or more handlers may apply their changes
 * successfully before one fails. Successful handlers are {@link #rollback(HomeUpdate) rolled back}, to allow them to
 * undo the changes they applied. <i>Handlers are <b>strongly</b> encouraged to implement rollback support.</i> Doing
 * so allows administrators to move the shared home directory back to its old location and start the system normally.
 * Handlers are rolled back in <i>reverse</i> order from how they are applied, and the handler that failed is <i>not</i>
 * rolled back.
 * <p>
 * Consider the following scenario:
 * <ul>
 *     <li>4 {@code HomeUpdateHandler}s have been registered</li>
 *     <li>Handler 1 {@link #apply(HomeUpdate) applies} <i>all</i> of its changes successfully</li>
 *     <li>Handler 2 {@link #apply(HomeUpdate) applies} <i>all</i> of its changes successfully</li>
 *     <li>Handler 3 {@link #apply(HomeUpdate) applies} <i>some</i> changes successfully, then one change fails and it
 *     throws a custom {@link ServiceException ServiceException}
 *     <ul>
 *         <li>Handler 3 is <i>not {@link #rollback(HomeUpdate) rolled back}</i></li>
 *         <li>Handler 2 is {@link #rollback(HomeUpdate) rolled back}</li>
 *         <li>Handler 1 is {@link #rollback(HomeUpdate) rolled back}</li>
 *     </ul>
 *     </li>
 *     <li>Handler 4 is <i>never {@link #apply(HomeUpdate) applied}</i></li>
 * </ul>
 * After the failure, administrators have two options:
 * <ol>
 *     <li>Fix the issue which caused the failure and restart the system still using the new location
 *     <ul>
 *         <li>{@code HomeUpdateHandler}s are invoked again to try and apply the update</li>
 *     </ul>
 *     </li>
 *     <li>Move the shared home directory back and restart the system using the old location
 *     <ul>
 *         <li><i>{@code HomeUpdateHandler}s are not invoked on startup</i>, because the shared home directory has
 *         not moved</li>
 *         <li>In this scenario, if Handler 1 or Handler 2 did not implement {@link #rollback(HomeUpdate) rollback
 *         support}, simply moving the shared home directory back to its old location would likely cause obscure
 *         failures because the updates that were applied mean some aspects of the system now expect the new location
 *         where others still expect the old one.</li>
 *     </ul>
 *     </li>
 * </ol>
 * In addition to simple path changes, where a shared home directory is moved to a new location on the same machine,
 * it is also possible that the operating system was changed as well. Implementations <i>must not</i> assume that the
 * old and new paths provided by the {@link HomeUpdate update} are both for the same operating system.
 *
 * <p>
 * Note: Due to change of the home directory structure, this class is updated to work on the shared home (where the data
 * directory will live from now on). When the directory structure is upgraded the associated {@link HomeUpdateHandler}
 * will be invoked to prepare the instance after the upgrade.
 *
 * @see HomeUpdate
 */
public interface HomeUpdateHandler {

    /**
     * Applies whatever changes are necessary to replace references to the old shared home directory location with
     * references to the new one.
     * <p>
     * Throwing a {@link ServiceException} indicates a required change could not be applied and will prevent the system
     * from being unlocked. Processing stops at the first exception, so any handlers after the one which throws will be
     * skipped.
     * <p>
     * If a {@link ServiceException} is thrown, its {@link ServiceException#getLocalizedMessage() localized message} is
     * shown to users. For any other {@code RuntimeException}, the standard message will be shown instead. Because the
     * message will be displayed to users, including unauthenticated users, it should not contain any sensitive data.
     * For example, file system paths should not be included. Instead, the message should contain a simple, high-level
     * summary of the failure and more comprehensive details should be written to the log for review by administrators.
     *
     * @param update provides the old and new shared home directory locations
     * @throws ServiceException if changes cannot be applied
     */
    void apply(@Nonnull HomeUpdate update) throws ServiceException;

    /**
     * Rolls back the {@link #apply(HomeUpdate) applied} changes, restoring references to the old shared home directory
     * location. Rolling back updates is intended to allow the system to be restarted with the old shared home directory
     * after updating to a new location fails.
     * <p>
     * This method is <i>only</i> invoked on handlers which apply successfully. If {@link #apply(HomeUpdate)} throws an
     * exception, this method <i>will not be called</i>. Handlers are rolled back in reverse order, so {@code A -> B ->
     * C} would roll back {@code B -> A} if handler C threw an exception.
     * <p>
     * Implementations are <i>encouraged</i> not to throw exceptions, as there is nothing more the system can do beyond
     * calling this method when {@link #apply(HomeUpdate) apply} fails. Failure cases should be handled internally and
     * logged, rather than being propagated to the system.
     *
     * @param update provides the old and new shared home directory locations
     */
    void rollback(@Nonnull HomeUpdate update);
}
