package com.atlassian.bitbucket.comment;

import javax.annotation.Nonnull;
import java.util.Iterator;

/**
 * An implementation of {@link CommentPropertyProvider} that flattens the list of the comments in the
 * {@link CommentPropertyContext context} such that subclasses that override
 * {@link #doProvideProperties(CommentPropertyContext)} have instant access to all nested comments without having
 * to go through the {@link Comment#getComments() reply tree} manually.
 *
 * @see CommentPropertyProvider
 */
public abstract class FlatteningCommentPropertyProvider implements CommentPropertyProvider {

    @Override
    public void provideProperties(@Nonnull CommentPropertyContext context) {
        doProvideProperties(flattenContext(context));
    }

    protected abstract void doProvideProperties(@Nonnull CommentPropertyContext context);

    private CommentPropertyContext flattenContext(final CommentPropertyContext original) {
        final Iterable<Comment> comments = CommentChain.of(original);
        return new CommentPropertyContext() {
            @Override
            public <T> T accept(@Nonnull CommentableVisitor<T> visitor) {
                return original.accept(visitor);
            }

            @Override
            public Iterator<Comment> iterator() {
                return comments.iterator();
            }

            @Override
            public void setProperty(@Nonnull Comment entity, @Nonnull String key, @Nonnull Object value) {
                original.setProperty(entity, key, value);
            }
        };
    }
}
