package com.atlassian.bitbucket.build.server.operations;

import com.atlassian.bitbucket.build.status.RepositoryBuildStatus;

import javax.annotation.Nonnull;
import java.net.URI;
import java.util.Optional;

/**
 * An interface for Bitbucket to interact with the remote CI tool in order to perform actions (e.g. run build) and
 * authorization.
 *
 * @since 7.8
 */
public interface PluginBuildServerClient {

    /**
     * Get and action by ID.
     *
     * Given the ID return the Action it corresponds to or {@link Optional#empty()} if the ID does not correspond to
     * any action this client is aware of. It is <strong>strongly</strong> advised to not issue any requests to
     * external systems as part of this method call. Authorisation and credential checks should <strong>not</strong> be
     * performed as part of this method call.
     *
     * @param buildStatus the build status that the action relates to
     * @param actionId the ID of the action that should be returned
     * @return the {@link BuildStatusAction} for the provided ID, or {@link Optional#empty()} if no action with the provided
     * ID is available for the provided build status
     */
    @Nonnull
    Optional<BuildStatusAction> getAction(@Nonnull RepositoryBuildStatus buildStatus, @Nonnull String actionId);

    /**
     * A collection containing the actions that can be performed by the currently authenticated user.
     *
     * @param buildStatus the build status to run the actions on.
     * @return a collection containing the allowed actions for the current user
     */
    @Nonnull
    BuildOperations getOperations(@Nonnull RepositoryBuildStatus buildStatus);

    /**
     * The URI that can be used for the current user to authorize Bitbucket to act with the requested build server on
     * their behalf. The URI should be returned regardless of the current user's need to authorize.
     *
     * @param callback the Web UI URI to redirect to after authorization is complete
     * @return the URL that can be used to provide the user the opportunity to authorize Bitbucket Server to
     *         access the build server on behalf of the user (i.e. typically OAuth style authorization)
     */
    @Nonnull
    URI getAuthorizationUrl(@Nonnull URI callback);

    /**
     * The URI that can be used for the current user to authorize Bitbucket to act with the requested build server on
     * their behalf. The URI should be returned regardless of the current user's need to authorize.
     *
     * @return the URI that can be used to provide the user the opportunity to authorize Bitbucket Server to
     *         access the build server on behalf of the user (i.e. typically OAuth style authorization)
     */
    @Nonnull
    URI getAuthorizationUrl();

    /**
     * Check whether or not Bitbucket has been authorized to act on this build server on the current user's behalf.
     *
     * This may require reaching out to the remote system to validate that credentials still work.
     *
     * @return {@code true} if the current user needs to authorize, or {@code false} if they are already authorized
     */
    boolean isAuthorizationRequired();

    /**
     * Start the action.
     *
     * Typically implementations wait for the remote system to <em>acknowledge</em> the request but not to complete the
     * requested action. No exceptions are expected from this, rather catch relevant exception and map them to relevant
     * fields in the {@link ActionResult}.
     *
     * @param request the request containing the information to perform the action
     * @return result of the action.
     */
    @Nonnull
    ActionResult performAction(@Nonnull BuildStatusActionRequest request);
}
