package com.atlassian.bitbucket.build.server;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Optional;

/**
 * Build Server Provider SPI.
 * <p>
 * Provides the ability for a plugin to be a "build server provider"; that is a plugin that provides the ability to
 * access and manage build jobs on a build/automation server.
 *
 * @since 7.8
 */
public interface PluginBuildServerProvider {

    /**
     * Get the {@link PluginBuildServer} for the given ID.
     *
     * @param id the ID of the build server to get
     * @return the associated {@link PluginBuildServer}, or {@link Optional#empty()} if there is no build server for the
     *         given ID
     */
    Optional<PluginBuildServer> getBuildServer(@Nonnull String id);

    /**
     * Given a request, identify the build server that made the request.
     * <p>
     * This is used to establish the concept of a trusted build status. A build status must be verified/trusted for
     * the purposes of certain functions such as:
     * <ul>
     *     <li>Presenting a list of downloadable build artifacts</li>
     *     <li>Permitting actions such as running a build from Bitbucket</li>
     * </ul>
     * Without establishing the authenticity of a build status, fraudulent build status reports may represent a
     * security risk. As such it is critical that that method only return a build server if, and only if, the
     * authenticity of the build status can be validated. <b>If authenticity cannot be validated then this method MUST
     * return {@link Optional#empty()}.</b>
     *
     * @param servletRequest the request for which the requester will be determined
     * @param requestBody    a map of all fields included in the build status request, fields of this map can be
     *                       used to verify the build server
     * @return An {@link Optional} containing the build server if the request can be validated as certainly originating
     *         from that build server. If the request cannot be validated as coming from a known/trusted build server this
     *         provider is responsible for, then {@link Optional#empty()} is returned.
     */
    @Nonnull
    Optional<PluginBuildServer> getBuildServer(@Nonnull HttpServletRequest servletRequest,
                                               @Nonnull Map<String, Object> requestBody);
}
