package com.atlassian.bitbucket.auth;

import com.atlassian.bitbucket.scm.BaseWeightedModuleDescriptor;
import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Implementations should be stateless and thread-safe, as a single instance will be used to service all authentication
 * requests.
 */
public interface SshAuthenticationHandler {

    /**
     * Invoked by the system when authentication is attempted over SSH. Implementations must
     * <ul>
     *     <li>
     *         return an {@link AuthenticationResult} if the user is authenticated successfully
     *     </li>
     *     <li>
     *         return {@code null} if the handler does not know how to authenticate the user from the request (for
     *         instance when a public key cannot be found in the system). By returning {@code null}, other
     *         {@link SshAuthenticationHandler handlers} will be attempted.
     *     </li>
     *     <li>
     *         throw an {@link AuthenticationException} if the provided authentication
     *         details are invalid (e.g. invalid username / public key, etc.). In this case, no other
     *         {@link SshAuthenticationHandler handlers} will be attempted.
     *     </li>
     * </ul>
     * All registered {@link SshAuthenticationHandler handlers} will be invoked in
     * {@link BaseWeightedModuleDescriptor weight order} until a handler successfully
     * authenticates the {@link ApplicationUser user}, or throws an {@link AuthenticationException}.
     * @param authenticationContext the authentication context that provides username, public key, remote address, etc
     * @return an {@link AuthenticationResult} if the handler was able to authenticate a user based on the request, or
     *         {@code null} if the handler opted out
     * @throws AuthenticationException when authentication fails.
     * @since 5.5
     */
    @Nullable
    AuthenticationResult performAuthentication(@Nonnull SshAuthenticationContext authenticationContext);
}
