package com.atlassian.bitbucket.auth;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

public class SimpleAuthentication implements Authentication {

    private final Map<String, Serializable> properties;
    private final ApplicationUser user;

    private SimpleAuthentication(Builder builder) {
        properties = builder.properties.build();
        user = builder.user;
    }

    @Nonnull
    public static Builder builder(@Nullable ApplicationUser user) {
        return new Builder(user);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        SimpleAuthentication that = (SimpleAuthentication) o;
        return Objects.equals(user, that.user) && Objects.equals(properties, that.properties);
    }

    @Nonnull
    @Override
    public Map<String, Serializable> getProperties() {
        return properties;
    }

    @Nonnull
    @Override
    public Optional<ApplicationUser> getUser() {
        return ofNullable(user);
    }

    @Override
    public int hashCode() {
        return Objects.hash(properties, user);
    }

    public static class Builder {

        private final ImmutableMap.Builder<String, Serializable> properties;
        private final ApplicationUser user;

        public Builder(@Nullable ApplicationUser user) {
            this.user = user;

            properties = ImmutableMap.builder();
        }

        @Nonnull
        public Authentication build() {
            return new SimpleAuthentication(this);
        }

        @Nonnull
        public Builder property(@Nonnull String key, @Nonnull Serializable value) {
            properties.put(requireNonNull(key, "key"), requireNonNull(value, "value"));
            return this;
        }

        @Nonnull
        public Builder properties(@Nonnull Map<String, Serializable> values) {
            properties.putAll(requireNonNull(values, "properties"));
            return this;
        }
    }
}
