package com.atlassian.bitbucket.auth;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.Map;

import static java.util.Objects.requireNonNull;

/**
 * Outcome of a successful {@link HttpAuthenticationHandler#performAuthentication(HttpAuthenticationContext) authentication}
 * operation.
 *
 * @since 5.5
 */
public class AuthenticationResult {

    private final Map<String, Serializable> properties;
    private final ApplicationUser user;

    private AuthenticationResult(Builder builder) {
        properties = builder.properties.build();
        user = builder.user;
    }

    /**
     * @return the authentication properties provided on authentication
     */
    @Nonnull
    public Map<String, Serializable> getProperties() {
        return properties;
    }

    @Nonnull
    public ApplicationUser getUser() {
        return user;
    }

    public static class Builder {

        private final ImmutableMap.Builder<String, Serializable> properties;
        private final ApplicationUser user;

        public Builder(@Nonnull ApplicationUser user) {
            this.user = requireNonNull(user, "user");

            properties = ImmutableMap.builder();
        }

        @Nonnull
        public AuthenticationResult build() {
            return new AuthenticationResult(this);
        }

        @Nonnull
        public Builder property(@Nonnull String key, @Nonnull Serializable value) {
            properties.put(requireNonNull(key, "key"), requireNonNull(value, "value"));
            return this;
        }

        @Nonnull
        public Builder properties(@Nonnull Map<String, Serializable> values) {
            properties.putAll(requireNonNull(values, "properties"));
            return this;
        }
    }
}
