'use strict';

var _require = require('lodash'),
    get = _require.get;

define('bitbucket-plugin-git-lfs/internal/feature/file-handlers/lfs-file-handler', ['jquery', 'lodash', 'bitbucket/util/navbuilder', 'bitbucket/util/server', 'bitbucket/util/state', 'bitbucket/internal/bbui/image-differ/image-differ', 'bitbucket/internal/feature/comments/comments', 'bitbucket/internal/feature/file-content/diff-view-options', 'bitbucket/internal/feature/file-content/diff-view-segment-types', 'bitbucket/internal/feature/file-content/handlers/diff-handler/diff-handler-internal', 'bitbucket/util/events', 'bitbucket/internal/util/time', 'bitbucket/internal/feature/file-content/request-diff', 'bitbucket/internal/feature/file-content/request-source', 'bitbucket/internal/model/file-change', 'bitbucket/internal/enums', 'bitbucket/internal/feature/file-content/file-content-modes'], function ($, _, nav, server, pageState, imageDiffer, comments, DiffViewOptions, DiffSegmentTypes, diffHandlerInternal, events, time, requestDiff, requestSource, FileChange, _ref, fileContentModes) {
    var ChangeTypes = _ref.ChangeTypes;

    var DIFF_HANDLER_ID = 'lfs-diff-handler';
    var SOURCE_HANDLER_ID = 'lfs-source-handler';

    var lfsMatchers = {
        version: /version https:\/\/git-lfs\.github\.com\/spec\/v(\d)/,
        oid: /oid sha256:([a-fA-F0-9]{64})/,
        size: /size ([0-9]*)/
    };

    var MAX_INLINE_IMAGE_SIZE = 10 * 1024 * 1024; //10MB

    var webImageContentTypes = {
        bmp: 'image/bmp',
        gif: 'image/gif',
        ico: 'image/x-icon',
        jpg: 'image/jpeg',
        jpeg: 'image/jpeg',
        png: 'image/png'
    };

    function enrichResponseObjects(paths, response, lockData) {
        return Object.keys(response).map(function (responseKey, i) {
            var responseObject = response[responseKey];
            var href = _.get(responseObject, 'actions.download.href');
            var path = paths[i];
            var extension = path.extension ? path.extension.toLowerCase() : '';
            var imageContentType = webImageContentTypes[extension];
            var extraProps = {
                fileName: path.name,
                extension: extension,
                href: href,
                inlineImage: !!(responseObject.size <= MAX_INLINE_IMAGE_SIZE && imageContentType),
                lock: lockData
            };

            return _.assign({}, responseObject, extraProps);
        });
    }

    function extractLFSMetaDataFromLines(lines) {
        var lfsMetaData = {};
        var lfsFields = Object.keys(lfsMatchers);

        function lineToMetaData(line, index) {
            var currentField = lfsFields[index];
            var matches = line.match(lfsMatchers[currentField]);

            if (matches) {
                return lfsMetaData[currentField] = matches[1];
            }

            return false;
        }

        return lines.length === 3 && lines.every(lineToMetaData) && lfsMetaData;
    }

    function extractLFSMetaDataFromSegments(segments) {
        var destinationLines = [];
        var sourceLines = [];

        segments.forEach(function (segment) {
            var segmentLines = _.map(segment.lines, 'line');

            switch (segment.type) {
                case DiffSegmentTypes.ADDED:
                    sourceLines.push.apply(destinationLines, segmentLines);
                    break;
                case DiffSegmentTypes.REMOVED:
                    destinationLines.push.apply(sourceLines, segmentLines);
                    break;
                default:
                    //CONTEXT
                    sourceLines.push.apply(sourceLines, segmentLines);
                    destinationLines.push.apply(destinationLines, segmentLines);
            }
        });

        var lfsMetadata = [destinationLines, sourceLines].map(extractLFSMetaDataFromLines);

        return { destinationMeta: lfsMetadata[0], sourceMeta: lfsMetadata[1] };
    }

    function getLFSLockInfo(paths) {
        var pathAndFileName = _.first(paths).components.join('/');
        var restConfig = {
            url: nav.rest('git-lfs').repository(pageState.getRepository()).addPathComponents('locks').withParams({ path: pathAndFileName }).build(),
            type: 'GET'
        };

        return server.rest(restConfig).then(function (response) {
            var lock = _.first(response.values);

            if (lock) {
                var lockedAt = time.format(lock['locked_at'], 'full');
                var lockData = {
                    author: lock.owner.name,
                    authorId: lock.owner.id,
                    date: lockedAt,
                    id: lock.id
                };

                events.trigger('bitbucket.internal.DO_NOT_USE.layout.files.lfs.locked', null, lockData);

                return lockData;
            }

            return lock;
        });
    }

    function generateImageURL(path, hash) {
        var imageURL = nav.repository(pageState.getRepository()).browse().path(path.components).withParams({ followLfs: true, raw: true, at: hash }).build();

        return imageURL;
    }

    function extractLFSData(lfsMetaDataObjects, paths, fromHash, toHash) {
        if (!lfsMetaDataObjects && Object.keys(lfsMetaDataObjects).length) {
            return reject();
        }

        var destinationMeta = lfsMetaDataObjects.destinationMeta,
            sourceMeta = lfsMetaDataObjects.sourceMeta;


        var path = paths[0] || paths[1];

        if (destinationMeta) {
            _.set(destinationMeta, 'actions.download.href', generateImageURL(path, toHash));
        }
        if (sourceMeta) {
            _.set(sourceMeta, 'actions.download.href', generateImageURL(path, fromHash));
        }

        return getLFSLockInfo(paths).then(function (lockData) {
            return enrichResponseObjects(paths, lfsMetaDataObjects, lockData);
        });
    }

    function handleLFSDiff(options) {
        if (options.isExcerpt) {
            return reject();
        }

        var fileChange = new FileChange(options.fileChange);
        var diffOptions = _.assign({}, options, {
            withComments: options.commentMode !== comments.commentMode.NONE
        });

        //Try and predict if this will be displayed as a side-by-side diff if it's not an LFS file
        //and adjust the diff options accordingly to ensure a cache hit if the diff is handled by diff-handler.js
        //Needs to match SBS selection logic in diff-handler.js
        if (DiffViewOptions.get('diffType') === 'side-by-side' && fileChange.getType() !== ChangeTypes.ADD && fileChange.getType() !== ChangeTypes.DELETE) {
            diffOptions = _.assign(diffOptions, {
                contextLines: diffHandlerInternal.infiniteContext,
                withComments: false
            });
        }

        return requestDiff(fileChange, diffOptions).then(function (data) {
            var diff = data.diff;

            if (!diff || diff.binary || !isLFSLikeDiff(diff)) {
                return reject();
            }

            var lfsMetaDataObjects = _.pickBy(extractLFSMetaDataFromSegments(diff.hunks[0].segments), Boolean);
            var paths = _.compact([diff.destination, diff.source]);

            if (lfsMetaDataObjects && Object.keys(lfsMetaDataObjects).length !== paths.length) {
                return reject();
            }

            return extractLFSData(lfsMetaDataObjects, paths, data.fromHash, data.toHash).then(renderDiffView.bind(null, options.$container, !diff.destination));
        }, function () {
            return reject();
        });
    }

    function handleLFSSource(options) {
        return requestSource(options.fileChange).then(function (data) {
            if (data.binary || data.start !== 0 || data.size !== 3) {
                return reject();
            }

            var lfsMetaDataObject = extractLFSMetaDataFromLines(_.map(data.lines, 'text'));

            if (!lfsMetaDataObject) {
                return reject();
            }

            var revision = get(options, 'fileChange.commitRange.untilRevision.id');

            var path = options.fileChange.path;

            return extractLFSData({ sourceMeta: lfsMetaDataObject }, [path], revision).then(renderSourceView.bind(null, options.$container, options.$toolbar));
        }, function () {
            return reject();
        });
    }

    function isLFSLikeDiff(diff) {
        //cheap heuristic
        var lfsDiffShape = {
            destinationLine: diff.destination ? 1 : 0, //for deleted files
            destinationSpan: diff.destination ? 3 : 0, //for deleted files
            sourceLine: diff.source ? 1 : 0, //for added files
            sourceSpan: diff.source ? 3 : 0, //for added files
            truncated: false
        };

        return !!(diff.hunks && diff.hunks.length === 1 && _.isMatch(diff.hunks[0], lfsDiffShape));
    }

    function reject(data) {
        return $.Deferred().reject(data);
    }

    function renderSourceView($container, $toolbar, lfsFiles) {
        var sourceFile = lfsFiles[0];

        $container.append(bitbucketPluginGitLfs.internal.fileHandlers.lfsFileViews.source({
            file: sourceFile
        }));

        $container.parent().addClass('lfs-file-content');

        $toolbar.find('.raw-view-link').attr('href', sourceFile.href).toggle(!sourceFile.error);

        return { handlerID: SOURCE_HANDLER_ID, lock: _.get(sourceFile, 'lock') };
    }

    function renderDiffView($container, isDelete, lfsFiles) {
        $container.append(bitbucketPluginGitLfs.internal.fileHandlers.lfsFileViews.diff({
            files: {
                //`New` is always the first file, except when the file has been deleted
                // prettier-ignore
                'new': !isDelete ? lfsFiles[0] : null,
                // prettier-ignore
                'old': !isDelete ? lfsFiles[1] : lfsFiles[0]
            }
        }));

        $container.parent().addClass('lfs-file-content');

        if (_.filter(lfsFiles, { inlineImage: true }).length === 2) {
            return imageDiffer.init($container.find('.binary-container')).then(function (imageDifferInfo) {
                imageDifferInfo.handlerID = DIFF_HANDLER_ID;

                return imageDifferInfo;
            });
        }

        return { handlerID: DIFF_HANDLER_ID };
    }

    return function handleLFSFile(options) {
        switch (options.contentMode) {
            case fileContentModes.SOURCE:
                return handleLFSSource(options);
            case fileContentModes.DIFF:
                return handleLFSDiff(options);
            default:
                return reject();
        }
    };
});