package com.atlassian.bitbucket.util;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.SortedMap;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

import static com.atlassian.bitbucket.util.MoreStreams.streamIterable;

/**
 * Provides support for pagination.
 * <p>
 * Pagination makes <em>no guarantee</em> that the next page will have a starting offset of {@link Page#getStart()} +
 * {@link Page#getLimit()}. Callers <em>must</em> use {@link Page#getNextPageRequest()} to navigate to the next page.
 *
 * @param <T> the type the entity
 */
public interface Page<T> {

    /**
     * Performs an action on each result in the page.
     *
     * @param action the action to apply to each result
     * @since 6.3
     */
    default void forEach(@Nonnull Consumer<? super T> action) {
        stream().forEach(action);
    }

    /**
     * @return {@code true} if there are no more results; otherwise, {@code false} if at least one more page,
     *         perhaps partially filled but not empty, of results is available
     */
    boolean getIsLastPage();

    /**
     * @return the original limit on the {@link PageRequest} that generated this page
     */
    int getLimit();

    /**
     * @return a request which can be used to retrieve the next page, which will be {@code null} if this was
     *         the {@link #getIsLastPage() last page}
     */
    @Nullable
    PageRequest getNextPageRequest();

    /**
     * Get a map of the page values mapped by their ordinal values. For filtered pages, the ordinals
     * are the ordinals in the underlying paged collection.
     *
     * @return values mapped by their ordinal value in the page
     */
    @Nonnull
    SortedMap<Integer, T> getOrdinalIndexedValues();

    /**
     * @return the number of results in the page
     */
    int getSize();

    /**
     * @return the offset into the overall result set the page starts at (0 based)
     */
    int getStart();

    /**
     * @return the page's results
     */
    @Nonnull
    Iterable<T> getValues();

    /**
     * @return a {@code Stream} over the page's results
     */
    @Nonnull
    default Stream<T> stream() {
        return streamIterable(getValues());
    }

    /**
     * Transforms the results on the page, producing a new {@code Page} with different {@link #getValues values} but
     * all other properties (e.g. {@link #getStart start} and {@link #getLimit limit}) unchanged.
     * <p>
     * <b>Implementation note</b>: Transformation is done <i>eagerly</i> to ensure the resulting {@code Page} does
     * not retain a reference to the original results from the source page. If a transform is used purely to apply
     * some set of side effects, consider using {@link #forEach} instead.
     *
     * @param transformFunction the transformer
     * @param <E> the target type
     * @return a new page with transformed results
     */
    @Nonnull
    <E> Page<E> transform(@Nonnull Function<? super T, ? extends E> transformFunction);
}
