package com.atlassian.bitbucket.topic;

import javax.annotation.Nonnull;
import java.io.Serializable;

/**
 * @since 5.3
 */
public interface Topic<T extends Serializable> {

    /**
     * @return the topic configuration
     */
    @Nonnull
    TopicSettings<T> getSettings();

    /**
     * Publishes a message to all (both local and on other cluster nodes) listeners of the topic.
     *
     * @param message the message
     */
    void publish(@Nonnull T message);

    /**
     * Subscribes the provided listener to the topic. The listener will receive callbacks for every message published
     * to the topic after the listener was subscribed.
     *
     * @param listener the listener that should receive callbacks when a message is published to the topic
     * @return the subscription ID, used to {@link #unsubscribe(String) unsubscribe}.
     */
    @Nonnull
    String subscribe(@Nonnull TopicListener<T> listener);

    /**
     * Unsubscribes a listener. The listener will receive no more messages from the topic.
     *
     * @param subscriptionId the subscription ID that was returned by {@link #subscribe(TopicListener)}
     * @return {@code true} if the listener was successfully unsubscribed, otherwise {@code false}
     */
    boolean unsubscribe(@Nonnull String subscriptionId);
}
