package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.MinimalRef;
import com.atlassian.bitbucket.repository.MinimalRefChange;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.util.Progress;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * Describes a callback for receiving streaming {@link RefChange ref changes}.
 * <p>
 * Implementations of this callback performing internal paging may return {@code false} from
 * {@link #onRefChange} to indicate no more ref changes are desired.
 *
 * @since 7.11
 */
public interface PushCallback {

    /**
     * Called after the final {@link #onRefChange ref change} has been streamed.
     * <p>
     * Note: If there were no ref changes, this method may be called immediately after {@link #onStart}
     * without any calls to {@link #onRefChange}.
     *
     * @param summary summarizes the request and the streamed changes
     * @throws IOException may be thrown by implementations which perform I/O
     */
    default void onEnd(@Nonnull PushSummary summary) throws IOException {
    }

    /**
     * Called when a ref failed to synchronize.
     *
     * @param ref the ref that failed
     * @throws IOException may be thrown by implementations which perform I/O
     */
    default void onFailedRef(@Nonnull MinimalRef ref) throws IOException {
    }

    /**
     * Called when there is a progress update for the operation.
     * <p>
     * Progress messages are taken directly from the output of the SCM and are not localized. Percentages are estimates
     * and are not guaranteed to reach <code>100</code> before {@link #onEnd} has been called.
     * Once {@link #onEnd} has been called, the progress should be assumed to be 100% complete.
     *
     * @param progress the message and percentage of the progress
     * @throws IOException may be thrown by implementations which perform I/O
     */
    default void onProgress(@Nonnull Progress progress) throws IOException {
    }

    /**
     * Provides a {@link RefChange ref change} to the callback for processing.
     *
     * @param refChange the ref change to process
     * @throws IOException may be thrown by implementations which perform I/O
     */
    default void onRefChange(@Nonnull MinimalRefChange refChange) throws IOException {
    }

    /**
     * Called before the <i>first</i> {@link #onRefChange ref change} is streamed.
     *
     * @param context provides details about the request for which ref changes are being streamed
     * @throws IOException may be thrown by implementations which perform I/O
     */
    default void onStart(@Nonnull PushContext context) throws IOException {
    }
}
