package com.atlassian.bitbucket.project;

import com.atlassian.bitbucket.avatar.AvatarSupplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkState;

/**
 * @see ProjectService#create(ProjectCreateRequest)
 */
public class ProjectCreateRequest extends AbstractProjectRequest {

    private final AvatarSupplier avatar;
    private final String avatarUri;

    private ProjectCreateRequest(Builder builder) {
        super(builder);

        avatar = builder.avatar;
        avatarUri = builder.avatarUri;
    }

    /**
     * @return a supplier containing an avatar image, or {@code null} to use a default avatar
     */
    @Nullable
    public AvatarSupplier getAvatar() {
        return avatar;
    }

    /**
     * @return a data URI containing a Base64-encoded avatar image, or {@code null} to use a default avatar
     */
    @Nullable
    public String getAvatarUri() {
        return avatarUri;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private String avatarUri;
        private AvatarSupplier avatar;

        public Builder() {
        }

        public Builder(Project project) {
            super(project);
        }

        /**
         * @param value a supplier containing an avatar image, or {@code null}
         * @return {@code this}
         */
        @Nonnull
        public Builder avatar(@Nullable AvatarSupplier value) {
            avatar = value;

            return self();
        }

        /**
         * The avatar data URI is required to contain Base64-encoded image data, and should be in the format:
         * <code>
         * data:(content type, e.g. image/png);base64,(data)
         * </code>
         *
         * @param value the avatar data URI or {@code null}
         * @return {@code this}
         */
        @Nonnull
        public Builder avatarUri(@Nullable String value) {
            avatarUri = value;

            return self();
        }

        @Nonnull
        public ProjectCreateRequest build() {
            // At _least_ one of these has to be null
            checkState(avatarUri == null || avatar == null, "Both an avatar and avatarUri cannot be provided");

            return new ProjectCreateRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
