package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.job.JobMessage;
import com.atlassian.bitbucket.job.JobMessageSeverity;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;

import static java.util.Optional.ofNullable;

/**
 * Request for {@link JobMessage job messages} from migration jobs.
 *
 * @since 5.13
 */
public class MigrationJobMessageSearchRequest {

    private final long jobId;
    private final Set<JobMessageSeverity> severities;
    private final String subject;

    private MigrationJobMessageSearchRequest(Builder builder) {
        this.jobId = builder.jobId;
        this.severities = builder.severities;
        this.subject = builder.subject;
    }

    public long getJobId() {
        return jobId;
    }

    @Nonnull
    public Set<JobMessageSeverity> getSeverities() {
        return severities;
    }

    @Nonnull
    public Optional<String> getSubject() {
        return ofNullable(subject);
    }

    public static class Builder extends BuilderSupport {

        private final long jobId;

        private EnumSet<JobMessageSeverity> severities;
        private String subject;

        public Builder(long jobId) {
            this.jobId = jobId;
            severities = EnumSet.noneOf(JobMessageSeverity.class);
        }

        @Nonnull
        public MigrationJobMessageSearchRequest build() {
            return new MigrationJobMessageSearchRequest(this);
        }

        @Nonnull
        public Builder severity(@Nullable JobMessageSeverity value) {
            addIf(java.util.Objects::nonNull, severities, value);
            return this;
        }

        @Nonnull
        public Builder severities(@Nonnull Iterable<JobMessageSeverity> values) {
            addIf(java.util.Objects::nonNull, severities, values);
            return this;
        }

        @Nonnull
        public Builder subject(@Nullable String value) {
            subject = value;
            return this;
        }
    }
}
