package com.atlassian.bitbucket.mesh;

import com.atlassian.bitbucket.validation.annotation.RequiredString;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;

import static java.util.Objects.requireNonNull;

/**
 * Represents a node that manages git repositories
 *
 * @since 8.0
 */
public interface MeshNode {

    /**
     * @return the (immutable) ID of the node
     */
    long getId();

    /**
     * @return when the last successful request was made to the node, or {@code null}
     */
    @Nullable
    Date getLastSeenDate();

    /**
     * @return the human-readable name of the node, used in the UI and logs
     */
    @Nonnull
    @RequiredString
    String getName();

    /**
     * @return the node's RPC ID, which is used to identify the node during requests
     */
    @Nonnull
    default String getRpcId() {
        return Long.toString(getId());
    }

    /**
     * @return the node's RPC URL, including the scheme and port
     */
    @Nonnull
    @RequiredString
    String getRpcUrl();

    /**
     * @return the node's current state
     * @since 8.4
     */
    @Nonnull
    State getState();

    /**
     * @return {@code true} if the node's {@link #getState() state} is {@link State#AVAILABLE}
     * @since 8.4
     */
    boolean isAvailable();

    /**
     * @return {@code true} if the node is currently offline; otherwise, {@code false}
     */
    boolean isOffline();

    /**
     * @return {@code true} if this instance is the sidecar node; otherwise, {@code false}
     */
    default boolean isSidecar() {
        return false;
    }

    /**
     * Enumerates the possible states for a given {@link MeshNode Mesh node}.
     *
     * @since 8.4
     */
    enum State {

        /**
         * Indicates that the node is available and that requests can be routed to it.
         */
        AVAILABLE(1, "Available"),
        /**
         * Indicates that the node is being removed from the cluster. All repository replicas located on the
         * node are being migrated to other remaining nodes if there are sufficient remaining Mesh nodes.
         */
        DELETING(2, "Deleting"),
        /**
         * Indicates that the node is disabled. All RPC requests have drained and no new requests are routed to
         * the node. The node can be safely taken down for maintenance when it's in this state.
         */
        DISABLED(3, "Disabled"),
        /**
         * Indicates that the node is currently draining its RPC requests. This is an intermediate state between
         * {@link #AVAILABLE} and {@link #DISABLED} when (temporarily) disabling a node for maintenance.
         */
        DRAINING(4, "Draining"),
        /**
         * Indicates that the node is currently offline.
         */
        OFFLINE(5, "Offline");

        private final int id;
        private final String statusMessage;

        State(int id, String statusMessage) {
            this.id = id;
            this.statusMessage = requireNonNull(statusMessage, "statusMessage");
        }

        /**
         * Retrieves a unique identifier for this state. Unlike the enum's ordinal or name, this value will never
         * change for a given entry.
         *
         * @return the state's unique identifier
         */
        public int getId() {
            return id;
        }

        /**
         * Retrieves the status message describing this state.
         *
         * @return the state's status message
         */
        @Nonnull
        public String getStatusMessage() {
            return statusMessage;
        }

        /**
         * Retrieves the state associated with the specified {@link #getId() ID}.
         *
         * @param id the ID to retrieve a {@code State} for
         * @return the state with the specified ID
         * @throws IllegalArgumentException if no state exists with the specified ID
         */
        public static State fromId(int id) {
            for (State value : values()) {
                if (value.getId() == id) {
                    return value;
                }
            }

            throw new IllegalArgumentException("No Mesh.State is associated with ID [" + id + "]");
        }
    }
}
