package com.atlassian.bitbucket.hook.script;

import com.atlassian.bitbucket.scope.Scope;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * Configures a {@link HookScript} to run for a given {@link Scope scope}, and defines the {@link #getTriggerIds
 * triggers} for which it should be invoked.
 * <p>
 * A single script can be configured multiple times for different scopes. In such cases, the configuration for the
 * <i>most specific scope</i> supersedes less specific ones. For example, if a script is configured at both project
 * and repository level, the repository-level configuration <i>overrides and replaces</i> the project-level config
 * for that repository; the two are not merged.
 * <p>
 * When a script is configured for a given scope, it is invoked according to its {@link HookScript#getType type}.
 * <i>Configuration cannot override type.</i> If a given script implements a {@link HookScriptType#PRE pre}-style
 * hook, it can only be run as a {@code PreRepositoryHook}. Configuration only controls what scopes a script is run
 * for, and for which triggers.
 *
 * @since 6.2
 */
public interface HookScriptConfig {

    /**
     * @return the scope for which the script is configured
     */
    @Nonnull
    Scope getScope();

    /**
     * @return the configured script
     */
    @Nonnull
    HookScript getScript();

    /**
     * @return a set of {@link com.atlassian.bitbucket.hook.repository.RepositoryHookTrigger#getId trigger IDs} for
     *         which the script should be invoked, with an empty denoting <i>all</i> triggers
     */
    @Nonnull
    Set<String> getTriggerIds();
}
