package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.content.FileEditHookRequest;
import com.atlassian.bitbucket.event.branch.BranchCreationHookRequest;
import com.atlassian.bitbucket.event.branch.BranchDeletionHookRequest;
import com.atlassian.bitbucket.event.tag.TagCreationHookRequest;
import com.atlassian.bitbucket.event.tag.TagDeletionHookRequest;
import com.atlassian.bitbucket.scm.MergeCommandParameters;
import com.atlassian.bitbucket.scm.ScmExtendedCommandFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Optional.empty;
import static java.util.Optional.of;

/**
 * Built-in trigger types for {@code repository-hook} {@link RepositoryHookRequest}s.
 *
 * @since 5.0
 */
public enum StandardRepositoryHookTrigger implements RepositoryHookTrigger {

    /**
     * @see BranchCreationHookRequest
     */
    BRANCH_CREATE("branch-create"),
    /**
     * @see BranchDeletionHookRequest
     */
    BRANCH_DELETE("branch-delete"),
    /**
     * @see FileEditHookRequest
     */
    FILE_EDIT("file-edit"),
    /**
     * Invoked when {@link ScmExtendedCommandFactory#merge(MergeCommandParameters)} is called. Note that this is a
     * low-level SCM operation and should not generally be intercepted.
     *
     * @see MergeHookRequest
     */
    MERGE("merge", true),
    /**
     * @see PullRequestMergeHookRequest
     */
    PULL_REQUEST_MERGE("pull-request-merge"),
    /**
     * @see RepositoryPushHookRequest
     */
    REPO_PUSH("push", true),
    /**
     * @see TagCreationHookRequest
     */
    TAG_CREATE("tag-create"),
    /**
     * @see TagDeletionHookRequest
     */
    TAG_DELETE("tag-delete"),
    /**
     * An unclassified / not recognized trigger
     */
    UNKNOWN("unknown");

    private final boolean abortOnFirstVeto;
    private final String id;

    StandardRepositoryHookTrigger(String id) {
        this(id, false);
    }

    StandardRepositoryHookTrigger(String id, boolean abortOnFirstVeto) {
        this.abortOnFirstVeto = abortOnFirstVeto;
        this.id = id;
    }

    @Nonnull
    public static Optional<StandardRepositoryHookTrigger> fromId(@Nullable String id) {
        for (StandardRepositoryHookTrigger trigger : values()) {
            if (trigger.id.equals(id)) {
                return of(trigger);
            }
        }

        return empty();
    }

    @Nonnull
    @Override
    public String getId() {
        return id;
    }

    @Override
    public boolean isAbortOnFirstVeto() {
        return abortOnFirstVeto;
    }
}
