package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.repository.*;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Basic implementation of {@link MergeHookRequest}
 *
 * @since 5.0
 */
public class SimpleMergeHookRequest extends AbstractRepositoryHookRequest implements MergeHookRequest {

    private final RepositoryRef fromRef;
    private final String mergeHash;
    private final String message;
    private final List<RefChange> refChanges;
    private final String strategyId;
    private final RepositoryRef toRef;

    private SimpleMergeHookRequest(Builder builder) {
        super(builder);

        fromRef = requireNonNull(builder.fromRef, "fromRef");
        toRef = requireNonNull(builder.toRef, "toRef");
        mergeHash = builder.mergeHash;
        message = builder.message;
        strategyId = builder.strategyId;
        if (mergeHash == null) {
            refChanges = Collections.emptyList();
        } else {
            refChanges = Collections.singletonList(new SimpleRefChange.Builder()
                    .from(toRef)
                    .toHash(mergeHash)
                    .type(RefChangeType.UPDATE)
                    .build());
        }
    }

    @Nonnull
    @Override
    public RepositoryRef getFromRef() {
        return fromRef;
    }

    @Nonnull
    @Override
    public Optional<String> getMergeHash() {
        return ofNullable(mergeHash);
    }

    @Nonnull
    @Override
    public Optional<String> getMessage() {
        return ofNullable(message);
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return refChanges;
    }

    /**
     * @since 5.5
     */
    @Nonnull
    @Override
    public Optional<String> getStrategyId() {
        return ofNullable(strategyId);
    }

    @Nonnull
    @Override
    public RepositoryRef getToRef() {
        return toRef;
    }

    @Override
    public boolean isCrossRepository() {
        return !Objects.equals(fromRef.getRepository(), toRef.getRepository());
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private RepositoryRef fromRef;
        private String mergeHash;
        private String message;
        private String strategyId;
        private RepositoryRef toRef;

        public Builder(@Nonnull Repository repository) {
            super(repository, StandardRepositoryHookTrigger.MERGE);
        }

        @Nonnull
        public SimpleMergeHookRequest build() {
            return new SimpleMergeHookRequest(this);
        }

        @Nonnull
        public Builder fromRef(@Nonnull RepositoryRef value) {
            fromRef = requireNonNull(value, "fromRef");
            return self();
        }

        @Nonnull
        public Builder mergeHash(@Nullable String value) {
            mergeHash = value;
            return self();
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            message = value;
            return self();
        }

        /**
         * @param value the new value
         * @return {@code this}
         * @since 5.5
         */
        @Nonnull
        public Builder strategyId(@Nullable String value) {
            strategyId = value;
            return self();
        }

        @Nonnull
        public Builder toRef(@Nonnull RepositoryRef value) {
            toRef = requireNonNull(value, "toRef");
            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
