package com.atlassian.bitbucket.event.annotation;

import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.audit.entity.AuditEvent;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.audit.AuditCategory;
import com.atlassian.bitbucket.audit.AuditEventConverter;
import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.event.api.EventPublisher;

import java.lang.annotation.*;

/**
 * Annotation for events that should be audited by the system.
 * <p>
 * Annotated actions are audited by being added to the audit log, database and potentially other
 * {@link AuditConsumer consumers}.
 * <p>
 * This annotation only affects types used as arguments in {@link EventPublisher#publish}. An example usage would be
 * creating a domain event which extends {@link ApplicationEvent} for a specific action and annotating the event with
 * {@link Auditable} to audit the action.
 *
 * @see AuditEvent
 * @see AuditEventConverter
 * @since 7.0
 */
@Documented
@Inherited
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface Auditable {

    /**
     * The I18n key of the action for the audited event. If this key is not defined in I18n properties, it is directly used
     * without translation. If this value is not provided, the name of the annotated class is used as the action for the
     * audit event.
     *
     * @return the action
     * @since 7.4
     */
    String action() default "";

    /**
     * The I18n key of the category for the audited event. If this key is not defined in I18n properties, it is directly
     * used without translation.
     * <p>
     * {@link AuditCategory} is a collection of system defined categories. It is recommended to use one of these
     * categories where possible.
     *
     * @return the category
     * @see AuditCategory
     */
    String category();

    /**
     * A converter for converting the annotated event to an {@link AuditEvent}.
     *
     * @return a converter to extract the details of the event to be audited
     */
    Class<? extends AuditEventConverter<?>> converter();

    /**
     * The {@link CoverageLevel coverage level} for the audited event.
     *
     * @return the coverage level
     * @see CoverageLevel
     */
    CoverageLevel coverageLevel();
}
