package com.atlassian.bitbucket.commit;

import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Locale;

import static java.util.Objects.requireNonNull;

/**
 * A flag for indicating how merge commits should be filtered.
 *
 * @since 4.8
 */
public enum CommitListMergeFilter {

    /**
     * Exclude merge commits.
     */
    EXCLUDE,
    /**
     * Include merge commits. This does not exclude non-merge commits.
     */
    INCLUDE,
    /**
     * Include <i>only</i> merge commits.
     */
    ONLY;

    /**
     * Coerces a string to a {@link CommitListMergeFilter} or returns the supplied default value if this is not possible.
     *
     * @param stringValue the string value of the commit list merge filter
     * @param defaultValue the default value to return if coercion is not possible
     * @return a corresponding {@link CommitListMergeFilter} instance
     */
    public static CommitListMergeFilter fromString(@Nonnull String stringValue, @Nullable CommitListMergeFilter defaultValue) {
        requireNonNull(stringValue, "stringValue");

        try {
            return valueOf(StringUtils.upperCase(stringValue, Locale.ROOT));
        } catch (IllegalArgumentException | NullPointerException ignored) {
            // Fall-through intentional
        }

        return defaultValue;
    }
}
