package com.atlassian.bitbucket.watcher;

import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * A base class for request objects used to interact with the {@link WatcherService} when referring to a specific
 * {@link Watcher watcher}.
 *
 * @since 5.10
 */
public class AbstractWatcherRequest {

    private final ApplicationUser user;
    private final Watchable watchable;

    protected AbstractWatcherRequest(@Nonnull AbstractBuilder<?, ?> builder) {
        this.user = builder.user;
        this.watchable = builder.watchable;
    }

    /**
     * If provided, determines which user is the subject of this request
     *
     * @return the user whose watch status is to be updated.
     */
    @Nonnull
    public Optional<ApplicationUser> getUser() {
        return ofNullable(user);
    }

    @Nonnull
    public Watchable getWatchable() {
        return watchable;
    }

    protected abstract static class AbstractBuilder<B, R> {

        private final Watchable watchable;

        private ApplicationUser user;

        public AbstractBuilder(@Nonnull Watchable watchable) {
            this.watchable = requireNonNull(watchable, "watchable");
        }

        @Nonnull
        public B user(@Nullable ApplicationUser value) {
            user = value;

            return self();
        }

        @Nonnull
        public abstract R build();

        @Nonnull
        protected abstract B self();
    }
}
