package com.atlassian.bitbucket.setting;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * A <i>typed</i> container for a generic map of settings.
 */
public interface Settings {

    /**
     * @param key of the setting to lookup
     * @return the {@code String} contained under this setting key or {@code null}
     * @throws ClassCastException is the underlying setting is not a {@code String}
     */
    @Nullable
    String getString(@Nonnull String key);

    /**
     * @param key of the setting to lookup
     * @param defaultValue the default value if the value isn't present
     * @return the {@code String} contained under this setting key or the default
     * @throws ClassCastException is the underlying setting is not a {@code String}
     */
    @Nonnull
    String getString(@Nonnull String key, @Nonnull String defaultValue);

    /**
     * @param key of the setting to lookup
     * @return the {@code Boolean} contained under this setting key, if the String value equals "true", or {@code null}
     */
    @Nullable
    Boolean getBoolean(@Nonnull String key);

    /**
     * @param key of the setting to lookup
     * @param defaultValue the default value if the value isn't present
     * @return the {@code boolean} contained under this setting key, if the String value equals "true", or the default
     */
    boolean getBoolean(@Nonnull String key, boolean defaultValue);

    /**
     * @param key of the setting to lookup
     * @return the {@code Integer} contained under this setting key, a parsed representation of the String, or {@code null}
     * @throws NumberFormatException if the resulting string is not a legal int
     */
    @Nullable
    Integer getInt(@Nonnull String key);

    /**
     * @param key of the setting to lookup
     * @param defaultValue the default value if the value isn't present
     * @return the {@code int} contained under this setting key, a parsed representation of the String, or the default
     * @throws NumberFormatException if the resulting string is not a legal int
     */
    int getInt(@Nonnull String key, int defaultValue);

    /**
     * @param key of the setting to lookup
     * @return the {@code Long} contained under this setting key, a parsed representation of the String, or {@code null}
     * @throws NumberFormatException if the resulting string is not a legal long
     */
    @Nullable
    Long getLong(@Nonnull String key);

    /**
     * @param key of the setting to lookup
     * @param defaultValue the default value if the value isn't present
     * @return the {@code long} contained under this setting key, a parsed representation of the String, or the default
     * @throws NumberFormatException if the resulting string is not a legal long
     */
    long getLong(@Nonnull String key, long defaultValue);

    /**
     * @param key of the setting to lookup
     * @return the {@code Double} contained under this setting key, a parsed representation of the String, or {@code null}
     * @throws NumberFormatException if the resulting string is not a legal double
     */
    @Nullable
    Double getDouble(@Nonnull String key);

    /**
     * @param key of the setting to lookup
     * @param defaultValue the default value if the value isn't present
     * @return the {@code double} contained under this setting key, a parsed representation of the String, or the default
     * @throws NumberFormatException if the resulting string is not a legal double
     */
    double getDouble(@Nonnull String key, double defaultValue);

    /**
     * @return a map representation of the settings
     */
    Map<String, Object> asMap();

}
