package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.scm.AbstractCommandParameters;

import javax.annotation.Nonnull;

import static com.atlassian.bitbucket.commit.AbstractCommitsRequest.UNLIMITED_MESSAGE_LENGTH;

/**
 * @since 6.0
 */
public class PullRequestCommitsCommandParameters extends AbstractCommandParameters {

    private final int maxMessageLength;
    private final boolean withMessages;

    private PullRequestCommitsCommandParameters(Builder builder) {
        maxMessageLength = builder.withMessages ? Math.max(UNLIMITED_MESSAGE_LENGTH, builder.maxMessageLength) : 0;
        //Only set withMessages to true if builder.withMessages was true _and_ the max length isn't 0
        withMessages = builder.withMessages && maxMessageLength != 0;
    }

    /**
     * @return the maximum length to read for {@link Commit#getMessage() commit messages}; if set, messages
     *         longer than this will be truncated; -1 indicates unlimited message length
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    /**
     * @return {@code true} if commit messages should be included; otherwise, {@code false} if they should be omitted
     */
    public boolean isWithMessages() {
        return withMessages;
    }

    public static class Builder {

        private int maxMessageLength;
        private boolean withMessages;

        public Builder() {
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
            withMessages = true;
        }

        @Nonnull
        public PullRequestCommitsCommandParameters build() {
            return new PullRequestCommitsCommandParameters(this);
        }

        /**
         * @param value the maximum length for included commit messages, where {@code 0} omits messages entirely
         *              and {@code -1} includes the full message regardless of length
         * @return {@code this}
         */
        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return this;
        }

        /**
         * @param value {@code true} to include commit messages in the output, up to the specified
         *              {@link #maxMessageLength(int) maximum length}; otherwise, {@code false} to
         *              omit messages entirely
         * @return {@code this}
         */
        @Nonnull
        public Builder withMessages(boolean value) {
            withMessages = value;

            return this;
        }
    }
}
