package com.atlassian.bitbucket.scm;

/**
 * Interface describing a watchdog which is used to manage a running process.
 * <p>
 * The watchdog helps prevent runaway processes by implementing an internal timer which kills the running process if
 * it is left untouched too long.
 *
 * @since 8.0
 */
public interface Watchdog {

    /**
     * Cancels the running process.
     */
    void cancel();

    /**
     * Retrieves a flag indicating whether the process has been canceled.
     *
     * @return {@code true} if the process has been canceled; otherwise, {@code false}
     */
    boolean isCanceled();

    /**
     * Resets the internal timer, which is used to track whether a process's handlers are idle or not and prevent
     * the idle timeout from terminating the process.
     * <p>
     * Note that the watchdog is reset automatically when a {@link CommandInputHandler input handler} writes to the
     * process stdin or when the {@link CommandOutputHandler output handler} or
     * {@link CommandErrorHandler error handler} reads from the process stdout or stderr.
     * <p>
     * As a result, <i>most</i> handlers do not need to explicitly reset the watchdog. This method exists to allow
     * handlers, which do lengthy processing in between accessing the {@code OutputStream}, to prevent idle timeouts
     * by manually resetting the watchdog.
     */
    void resetWatchdog();
}
