package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.FileContext;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;

public class FileCommandParameters extends AbstractCommitCommandParameters {

    private final boolean annotated;
    private final int maxLineLength;

    private FileCommandParameters(Builder builder) {
        super(builder);

        if (StringUtils.isBlank(getCommitId())) {
            throw new IllegalStateException("A commit ID must provided when retrieving file contents");
        }
        if (StringUtils.isBlank(getPath())) {
            throw new IllegalStateException("A path must be provided when retrieving file contents");
        }
        maxLineLength = builder.maxLineLength;
        annotated = builder.annotated;
    }

    public int getMaxLineLength() {
        return maxLineLength;
    }

    public boolean isAnnotated() {
        return annotated;
    }

    /**
     * Assembles a {@link FileContext} from these parameters.
     *
     * @return a {@link FileContext} describing these parameters
     */
    @Nonnull
    public FileContext toContext() {
        return new FileContext.Builder(getCommitId())
                .maxLineLength(getMaxLineLength())
                .build();
    }

    public static class Builder extends AbstractCommitParametersBuilder<Builder> {

        private boolean annotated;
        private int maxLineLength;

        public Builder() {
        }

        @Nonnull
        public Builder annotate(boolean value) {
            annotated = value;

            return this;
        }

        @Nonnull
        public FileCommandParameters build() {
            return new FileCommandParameters(this);
        }

        @Nonnull
        public Builder maxLineLength(int value) {
            maxLineLength = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
