package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.ArchiveFormat;
import com.atlassian.bitbucket.content.ArchiveRequest;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Describes a commit to {@link ScmExtendedCommandFactory#archive archive}.
 * <p>
 * In addition to specifying the commit to archive, callers can select from a {@link ArchiveFormat standard set} of
 * archive formats. Any SCM which supports {@link ScmFeature#ARCHIVE archiving} repositories is <i>required</i> to
 * support <i>all</i> of the standard formats.
 * <p>
 * For finer control over the content of the archive, callers can also specify {@link #getPaths() paths} to filter
 * by, and a {@link #getPrefix() prefix} to prepend to the archive's contents. SCMs <i>should</i> support both of
 * those settings if possible, but are <i>not required</i> to do so.
 *
 * @since 5.1
 */
public class ArchiveCommandParameters extends AbstractCommandParameters {

    private final String commitId;
    private final ArchiveFormat format;
    private final Set<String> paths;
    private final String prefix;

    private ArchiveCommandParameters(Builder builder) {
        commitId = builder.commitId;
        format = builder.format;
        paths = builder.paths.build();
        prefix = StringUtils.appendIfMissing(builder.prefix, "/"); //Has no effect if prefix is null
    }

    /**
     * @return the commit to archive, which may be a ref name or commit hash
     */
    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    /**
     * @return the format to stream the archive in
     */
    @Nonnull
    public ArchiveFormat getFormat() {
        return format;
    }

    /**
     * @return a set of paths to filter the archive by, which may be empty to include all paths
     */
    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    /**
     * @return a prefix to apply to all entries in the archive, which may be {@code empty()} but not {@code null}
     */
    @Nonnull
    public Optional<String> getPrefix() {
        return ofNullable(prefix);
    }

    public static class Builder {

        private final String commitId;
        private final ImmutableSet.Builder<String> paths;

        private ArchiveFormat format;
        private String prefix;

        public Builder(@Nonnull ArchiveRequest request) {
            this(requireNonNull(request, "request").getCommitId());

            format(request.getFormat())
                    .paths(request.getPaths())
                    .prefix(request.getPrefix().orElse(null));
        }

        public Builder(@Nonnull String commitId) {
            this.commitId = requireNonNull(commitId, "commitId");

            format = ArchiveFormat.ZIP;
            paths = ImmutableSet.builder();
        }

        @Nonnull
        public ArchiveCommandParameters build() {
            return new ArchiveCommandParameters(this);
        }

        @Nonnull
        public Builder format(@Nonnull ArchiveFormat value) {
            format = requireNonNull(value, "format");

            return this;
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return this;
        }

        @Nonnull
        public Builder prefix(@Nullable String value) {
            prefix = StringUtils.trimToNull(value);

            return this;
        }
    }
}
