package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

public abstract class AbstractRepositoryRefsRequest {

    private final boolean boostMatches;
    private final String filterText;
    private final RefOrder order;
    private final Repository repository;

    protected AbstractRepositoryRefsRequest(AbstractBuilder<?, ?> builder) {
        boostMatches = builder.boostMatches;
        filterText = builder.filterText;
        order = builder.order;
        repository = builder.repository;
    }

    /**
     * @return optional text for filtering returned refs
     */
    @Nullable
    public String getFilterText() {
        return filterText;
    }

    /**
     * @return identifier to help decide to boost search results with exact and prefix matching
     * @since 7.2
     */
    public boolean isBoostMatches() {
        return boostMatches;
    }

    /**
     * @return the ordering to apply when listing refs; if {@code null}, the order is determined by the SCM
     */
    @Nullable
    public RefOrder getOrder() {
        return order;
    }

    /**
     * @return the repository to retrieve refs from
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder<B, R>, R extends AbstractRepositoryRefsRequest>
            extends BuilderSupport {

        private final Repository repository;
        private boolean boostMatches;
        private String filterText;
        private RefOrder order;

        protected AbstractBuilder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");
        }

        /**
         * @since 7.2
         */
        @Nonnull
        public B boostMatches(boolean value) {
            boostMatches = value;

            return self();
        }

        @Nonnull
        public abstract R build();

        @Nonnull
        public B filterText(@Nullable String value) {
            filterText = value;

            return self();
        }

        @Nonnull
        public B order(@Nullable RefOrder value) {
            order = value;

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
