package com.atlassian.bitbucket.pull;

import javax.annotation.Nullable;

/**
 * Allows retrieving {@link PullRequest pull requests} by their {@link PullRequest#getToRef() target}
 * {@link PullRequestRef#getRepository() repository} and {@link PullRequest#getId() ID}.
 * <p>
 * This is a simplification of the {@link PullRequestService}, intended to be used in cases where a component needs to
 * be able to look up pull requests but doesn't need the full service functionality. Using this interface instead makes
 * such components easier to test.
 * <b>Plugin developers</b>: Plugins cannot autowire their components when importing both {@code pullRequestSupplier}
 * and {@link PullRequestService}. In general, if a plugin needs {@link PullRequestService} methods, it should
 * <i>only</i> use the {@link PullRequestService}. This interface should be used when a plugin only needs to retrieve
 * pull requests.
 */
public interface PullRequestSupplier {

    /**
     * Retrieves a pull request by its ID, within the specified repository.
     *
     * @param repositoryId  the ID of the repository to which the pull request will be merged
     * @param pullRequestId the ID of the pull request within the repository
     * @return the pull request found or null if there is no pull request that matches the given id
     */
    @Nullable
    PullRequest getById(int repositoryId, long pullRequestId);
}
