package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * @since 5.13
 */
public class RepositorySelector {

    private final String projectKey;
    private final String slug;

    private RepositorySelector(Builder builder) {
        projectKey = builder.projectKey;
        slug = builder.slug;
    }

    @Nonnull
    public static RepositorySelector of(@Nonnull String projectKey, @Nonnull String slug) {
        return new Builder(projectKey, slug).build();
    }

    /**
     * @since 8.0
     */
    @Nonnull
    public static RepositorySelector of(@Nonnull Repository repository) {
        requireNonNull(repository, "repository");

        return new Builder(repository.getProject().getKey(), repository.getSlug()).build();
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof RepositorySelector) {
            RepositorySelector p = (RepositorySelector) o;

            return Objects.equals(projectKey, p.projectKey) && Objects.equals(slug, p.slug);
        }

        return false;
    }

    @Nonnull
    public String getProjectKey() {
        return projectKey;
    }

    @Nonnull
    public String getSlug() {
        return slug;
    }

    @Override
    public int hashCode() {
        return Objects.hash(projectKey, slug);
    }

    public static final class Builder {

        private final String projectKey;
        private final String slug;

        public Builder(@Nonnull String projectKey, @Nonnull String slug) {
            this.projectKey = requireNonNull(projectKey, "projectKey");
            this.slug = requireNonNull(slug, "slug");

            if (projectKey.equals("*") && slug.equals("*")) {
                return;
            }
            if (projectKey.equals("*") && !slug.equals("*")) {
                throw new IllegalArgumentException(String.format(
                        "Invalid repository slug: '%s'. Repository slug should be '*', if project key is '*'.", slug));
            }
            if (projectKey.contains("*")) {
                throw new IllegalArgumentException(String.format(
                        "Partial matches are not supported in project key: '%s'.", projectKey));
            }
            if (!slug.equals("*") && slug.contains("*")) {
                throw new IllegalArgumentException(String.format(
                        "Partial matches are not supported in repository slug: '%s'.", slug));
            }
        }

        @Nonnull
        public RepositorySelector build() {
            return new RepositorySelector(this);
        }
    }
}
