package com.atlassian.bitbucket.hook.repository;

import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Describes the outcome of a {@code repository-hook}
 * {@link RepositoryHookService#preUpdate(RepositoryHookRequest) preUpdate} invocation.
 *
 * @since 5.0
 */
public class RepositoryHookResult {

    private static final RepositoryHookResult ACCEPTED = new RepositoryHookResult();

    private final List<RepositoryHookVeto> vetoes;

    private RepositoryHookResult() {
        vetoes = Collections.emptyList();
    }

    private RepositoryHookResult(@Nonnull Builder builder) {
        vetoes = builder.vetoes.build();
    }

    /**
     * @return result that accepts the proposed request
     */
    @Nonnull
    public static RepositoryHookResult accepted() {
        return ACCEPTED;
    }

    /**
     * @param summaryMessage a short summary of the reason the request is rejected
     * @param detailMessage a detailed description of the reason for rejecting the request
     * @return result that rejects the proposed request
     */
    @Nonnull
    public static RepositoryHookResult rejected(@Nonnull String summaryMessage, @Nonnull String detailMessage) {
        return new RepositoryHookResult.Builder()
                .veto(summaryMessage, detailMessage)
                .build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RepositoryHookResult result = (RepositoryHookResult) o;
        return Objects.equals(vetoes, result.vetoes);
    }

    /**
     * @return a list of vetoes that describe why the request was rejected. Empty if the request was accepted.
     */
    @Nonnull
    public List<RepositoryHookVeto> getVetoes() {
        return vetoes;
    }

    @Override
    public int hashCode() {
        return Objects.hash(vetoes);
    }

    /**
     * @return {@code true} if the request was accepted
     */
    public boolean isAccepted() {
        return vetoes.isEmpty();
    }

    /**
     * @return {@code true} if the request was rejected
     */
    public boolean isRejected() {
        return !isAccepted();
    }

    public static class Builder {

        private final ImmutableList.Builder<RepositoryHookVeto> vetoes;

        private boolean rejected;

        public Builder() {
            vetoes = ImmutableList.builder();
        }

        public Builder(@Nonnull RepositoryHookResult result) {
            vetoes = ImmutableList.builder();
            vetoes.addAll(requireNonNull(result, "result").getVetoes());
            rejected = result.isRejected();
        }

        /**
         * Includes all {@link #getVetoes() vetoes} from another result
         *
         * @param value the other result to include
         * @return {@code this}
         */
        @Nonnull
        public Builder add(@Nullable RepositoryHookResult value) {
            if (value != null) {
                vetoes.addAll(value.getVetoes());
                rejected |= value.isRejected();
            }
            return this;
        }

        /**
         * @return the result
         */
        @Nonnull
        public RepositoryHookResult build() {
            return rejected ? new RepositoryHookResult(this) : ACCEPTED;
        }

        public boolean isRejected() {
            return rejected;
        }

        /**
         * @param summaryMessage a short summary of the reason the request is rejected
         * @param detailedMessage a detailed description of the reason for rejecting the request
         * @return {@code this}
         */
        @Nonnull
        public Builder veto(@Nonnull String summaryMessage, @Nonnull String detailedMessage) {
            vetoes.add(new SimpleRepositoryHookVeto(summaryMessage, detailedMessage));
            rejected = true;
            return this;
        }
    }
}
