package com.atlassian.bitbucket.event.user;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.UserAdminService;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Raised when personally identifying user data is erased for {@link #getOriginalUsername() username}.
 * <p>
 * Clients that store personally identifying user data should implement a {@code UserErasureModuleDescriptor} with a
 * corresponding handler to erase this data.
 * <p>
 * This event is internally audited with a {@link CoverageLevel#BASE} coverage level.
 *
 * @see UserAdminService#eraseUser(String)
 * @since 5.16
 */
@AsynchronousPreferred
@EventName("stash.user.erasure.success")
@TransactionAware
public class UserErasedEvent extends ApplicationEvent {

    private final long duration;
    private final String originalUsername;
    private final ApplicationUser updatedUser;

    public UserErasedEvent(@Nonnull Object source, long duration, @Nonnull String originalUsername,
                           @Nonnull ApplicationUser updatedUser) {
        super(source);

        requireNonNull(originalUsername, "originalUsername");
        requireNonNull(updatedUser, "updatedUser");

        this.duration = duration;
        this.originalUsername = originalUsername;
        this.updatedUser = updatedUser;
    }

    /**
     * @return processing duration of the user erasure, in milliseconds
     */
    public long getDuration() {
        return duration;
    }

    /**
     * @return the original username of the erased user
     */
    @Nonnull
    public String getOriginalUsername() {
        return originalUsername;
    }

    /**
     * @return the user with the anonymized name
     */
    @Nonnull
    public ApplicationUser getUpdatedUser() {
        return updatedUser;
    }
}
