package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.pull.PullRequestParticipant;
import com.atlassian.bitbucket.pull.PullRequestParticipantStatus;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Base class for events raised when a {@link PullRequestParticipant participant} updates its
 * {@link PullRequestParticipantStatus status} on a {@link PullRequest pull request}.
 */
public class PullRequestParticipantStatusUpdatedEvent extends PullRequestEvent {

    private static final Set<PullRequestAction> STATUS_ACTIONS = ImmutableSet.of(
            PullRequestAction.APPROVED,
            PullRequestAction.REVIEWED,
            PullRequestAction.UNAPPROVED);

    private final PullRequestParticipant participant;
    private final PullRequestParticipantStatus previousStatus;

    /**
     * Constructs a new {@code PullRequestEvent}. <i>All parameters are required.</i>
     *
     * @param source      the component raising the event
     * @param pullRequest the affected pull request
     * @param action      the action performed on the pull request
     * @param participant the participant whose status has been updatedStatus
     * @param previousStatus    the participant status before the update
     */
    protected PullRequestParticipantStatusUpdatedEvent(@Nonnull Object source,
                                                       @Nonnull PullRequest pullRequest,
                                                       @Nonnull PullRequestAction action,
                                                       @Nonnull PullRequestParticipant participant,
                                                       @Nonnull PullRequestParticipantStatus previousStatus) {
        super(source, pullRequest, action);

        checkArgument(STATUS_ACTIONS.contains(action), "The action provided must be approval-related");

        this.participant = requireNonNull(participant, "participant");
        this.previousStatus = requireNonNull(previousStatus, "previousStatus");
    }

    @Nonnull
    public PullRequestParticipant getParticipant() {
        return participant;
    }

    @Nonnull
    public PullRequestParticipantStatus getPreviousStatus() {
        return previousStatus;
    }
}
