package com.atlassian.bitbucket.event.mirror;

import com.atlassian.bitbucket.repository.MinimalRef;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.mirror.RepositorySynchronizationType;
import com.atlassian.bitbucket.util.NamedLink;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;

import static java.util.Objects.requireNonNull;

/**
 * This event signals that a {@link Repository} has been synchronized on a mirror
 *
 * @since 5.0
 */
public class RepositoryMirrorSynchronizedEvent extends AbstractRepositoryMirrorEvent {

    private final Collection<NamedLink> cloneLinks;
    private final Collection<MinimalRef> failedRefs;
    private final Collection<RefChange> refChanges;
    private final boolean refLimitExceeded;
    private final RepositorySynchronizationType syncType;

    private RepositoryMirrorSynchronizedEvent(Builder builder) {
        super(builder);
        cloneLinks = builder.cloneLinks;
        failedRefs = builder.failedRefs;
        refChanges = builder.refChanges;
        refLimitExceeded = builder.refLimitExceeded;
        syncType = builder.syncType;
    }

    /**
     * @return a collection of links that can be used to clone the repository from this mirror
     * @since 6.7
     */
    @Nonnull
    public Collection<NamedLink> getCloneLinks() {
        return cloneLinks;
    }

    /**
     * @return a collection of refs which failed to update on the mirror
     */
    @Nonnull
    public Collection<MinimalRef> getFailedRefs() {
        return failedRefs;
    }

    /**
     * @return a collection of refs successfully updated on the mirror
     */
    @Nonnull
    public Collection<RefChange> getRefChanges() {
        return refChanges;
    }

    /**
     * @return the type of synchronization that was performed
     * @since 6.7
     */
    @Nullable
    public RepositorySynchronizationType getSyncType() {
        return syncType;
    }

    /**
     * @return true if amount of refs exceeded its limit which results in an empty list of ref changes; false otherwise
     * @since 6.7
     */
    public boolean isRefLimitExceeded() {
        return refLimitExceeded;
    }

    public static class Builder extends AbstractRepositoryMirrorEvent.Builder<Builder, RepositoryMirrorSynchronizedEvent> {

        private Collection<NamedLink> cloneLinks;
        private Collection<MinimalRef> failedRefs;
        private Collection<RefChange> refChanges;
        private boolean refLimitExceeded;
        private RepositorySynchronizationType syncType;

        @Nonnull
        @Override
        public RepositoryMirrorSynchronizedEvent build() {
            validate();
            return new RepositoryMirrorSynchronizedEvent(this);
        }

        @Nonnull
        public Builder cloneLinks(@Nonnull Collection<NamedLink> value) {
            cloneLinks = requireNonNull(value, "cloneLinks");
            return this;
        }

        @Nonnull
        public Builder failedRefs(@Nonnull Collection<MinimalRef> value) {
            failedRefs = requireNonNull(value, "failedRefs");
            return this;
        }

        @Nonnull
        public Builder refChanges(@Nonnull Collection<RefChange> value) {
            refChanges = requireNonNull(value, "refChanges");
            return this;
        }

        @Nonnull
        public Builder refLimitExceeded(boolean value) {
            refLimitExceeded = value;
            return this;
        }

        @Nonnull
        public Builder syncType(RepositorySynchronizationType value) {
            syncType = value;
            return this;
        }

        @Override
        protected void validate() {
            super.validate();
            if (cloneLinks == null) {
                throw new IllegalArgumentException("cloneLinks is required");
            }
            if (failedRefs == null) {
                throw new IllegalStateException("failedRefs is required");
            }
            if (refChanges == null) {
                throw new IllegalStateException("refChanges is required");
            }
            if (refLimitExceeded && !refChanges.isEmpty()) {
                throw new IllegalStateException("When ref limit is exceeded the list of ref changes should be empty");
            }
        }
    }
}