package com.atlassian.bitbucket.event.hook;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.setting.Settings;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Fired when a repository hook's settings are changed.
 * <p>
 * This event is internally audited with {@link CoverageLevel#BASE} level.
 */
@AsynchronousPreferred
@TransactionAware
@EventName("stash.repository.hook.settings.changed")
public class RepositoryHookSettingsChangedEvent extends RepositoryHookEvent {

    private final Settings oldSettings;
    private final Settings settings;

    /**
     * @param source            the object on which the event initially occurred
     * @param repositoryHookKey the key of the repository hook
     * @param scope             scope that the hook event was created for
     * @param settings          the settings that apply for the hook
     * @param oldSettings       the previous settings that were applied for the hook
     * @since 7.4
     */
    public RepositoryHookSettingsChangedEvent(@Nonnull Object source, @Nonnull Scope scope,
                                              @Nonnull String repositoryHookKey, @Nonnull Settings settings,
                                              @Nonnull Settings oldSettings) {
        super(source, repositoryHookKey, scope);
        this.settings = requireNonNull(settings, "settings");
        this.oldSettings = requireNonNull(oldSettings, "oldSettings");
    }

    @Nullable
    public Settings getOldSettings() {
        return oldSettings;
    }

    @Nonnull
    public Settings getSettings() {
        return settings;
    }
}
