package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Describes a request to format a patch of a given repository for a given commit or commit range.
 *
 * @since 6.7
 */
public class PatchRequest {

    private final boolean allAncestors;
    private final Repository repository;
    private final Repository secondaryRepository;
    private final String sinceId;
    private final String untilId;

    private PatchRequest(Builder builder) {
        allAncestors = builder.allAncestors;
        repository = builder.repository;
        secondaryRepository = builder.secondaryRepository;
        sinceId = builder.sinceId;
        untilId = builder.untilId;

        if (secondaryRepository != null &&
                ObjectUtils.notEqual(repository.getHierarchyId(), secondaryRepository.getHierarchyId())) {
            //This is a developer error; it should not be possible for a user to ever trigger this. Given that,
            //this is not internationalised (additionally, it'd be pretty difficult to do so anyway...)
            throw new IllegalStateException(
                    secondaryRepository.getProject().getKey() + "/" + secondaryRepository.getSlug() +
                            " is not from the same hierarchy as " +
                            repository.getProject().getKey() + "/" + repository.getSlug() +
                            "; patch may only be streamed between repositories from the same hierarchy.");
        }
    }

    /**
     * @return the repository
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * Retrieves the <b>secondary</b> repository from which to generate the patch. Commits in this repository may only
     * be identified <i>by hash</i>. Any branch or tag names used will <i>always</i> be resolved using the
     * {@link #getRepository() primary repository}.
     *
     * @return a secondary repository from which to generate the patch
     */
    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    /**
     * Retrieves the base revision from which to generate the patch. This is only applicable when
     * {@link #isAllAncestors()} is false. If omitted the patch will represent one single commit, the
     * {@link #getUntilId() until ID}.
     *
     * @return the base revision from which to generate the patch
     */
    @Nullable
    public String getSinceId() {
        return sinceId;
    }

    /**
     * @return the target revision from which to generate the patch
     */
    @Nonnull
    public String getUntilId() {
        return untilId;
    }

    /**
     * @return whether or not to generate a patch which includes all ancestors of {@link #getUntilId() until ID}.
     * If true, the value provided by {@link #getSinceId() since ID} is ignored
     */
    public boolean isAllAncestors() {
        return allAncestors;
    }

    public static class Builder extends BuilderSupport {

        private final Repository repository;
        private final String untilId;
        private boolean allAncestors;
        private Repository secondaryRepository;
        private String sinceId;

        public Builder(@Nonnull Repository repository, @Nonnull String untilId) {
            this.repository = requireNonNull(repository, "repository");
            this.untilId = requireNonBlank(untilId, "untilId");
        }

        @Nonnull
        public Builder allAncestors(boolean value) {
            allAncestors = value;

            return this;
        }

        @Nonnull
        public PatchRequest build() {
            return new PatchRequest(this);
        }

        @Nonnull
        public Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }

        @Nonnull
        public Builder sinceId(@Nullable String value) {
            sinceId = StringUtils.trimToNull(value);

            return this;
        }
    }
}
