package com.atlassian.bitbucket.compare;

import com.atlassian.bitbucket.content.AbstractDiffRequest;
import com.atlassian.bitbucket.content.DiffRequest;
import com.atlassian.bitbucket.pull.PullRequestDiffRequest;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * A diff request to compare two refs or commits.
 *
 * @see DiffRequest
 * @see PullRequestDiffRequest
 * @see CompareRef
 */
public class CompareDiffRequest extends AbstractDiffRequest {

    private final CompareRef fromRef;
    private final CompareRef toRef;

    private CompareDiffRequest(Builder builder) {
        super(builder);

        fromRef = builder.fromRef;
        toRef = builder.toRef;
    }

    /**
     * @return the {@link CompareRef ref} serving as the tip of the changes to compare
     */
    @Nonnull
    public CompareRef getFromRef() {
        return fromRef;
    }

    /**
     * @return the {@link CompareRef ref} serving as the baseline for those changes
     */
    @Nonnull
    public CompareRef getToRef() {
        return toRef;
    }

    public static class Builder extends AbstractBuilder<Builder, CompareDiffRequest> {

        private CompareRef fromRef;
        private CompareRef toRef;

        @Nonnull
        @Override
        public CompareDiffRequest build() {
            checkArgument(fromRef != null && toRef != null, "Both fromRef and toRef are required.");

            return new CompareDiffRequest(this);
        }

        @Nonnull
        public Builder fromRef(CompareRef value) {
            fromRef = requireNonNull(value, "fromRef");

            return self();
        }

        @Nonnull
        public Builder toRef(CompareRef value) {
            toRef = requireNonNull(value, "toRef");

            return self();
        }

        @Nonnull
        protected Builder self() {
            return this;
        }
    }

}
