package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

/**
 * Describes a request for a single commit. If no {@link #getPath path} is provided, the specified {@link #getCommitId
 * commit} will be the one returned. If a {@link #getPath path} <i>is</i> provided, the commit returned will be the
 * first commit in the <i>ancestry</i> of the specified {@link #getCommitId commit} which modifies that path.
 */
public class CommitRequest extends AbstractCommitRequest {

    private final Set<String> propertyKeys;

    private CommitRequest(Builder builder) {
        super(builder);

        propertyKeys = builder.propertyKeys.build();
    }

    /**
     * @return a set of {@link Commit#getProperties() property keys} to load, which may be empty but never {@code null}
     */
    @Nonnull
    public Set<String> getPropertyKeys() {
        return propertyKeys;
    }

    public static class Builder extends AbstractCommitRequestBuilder<Builder> {

        private final ImmutableSet.Builder<String> propertyKeys;

        public Builder(@Nonnull Repository repository, @Nonnull String commitId) {
            super(repository, commitId);

            propertyKeys = ImmutableSet.builder();
        }

        @Nonnull
        public CommitRequest build() {
            return new CommitRequest(this);
        }

        /**
         * @since 4.5
         */
        @Nonnull
        @Override
        public Builder maxMessageLength(int value) {
            return super.maxMessageLength(value);
        }

        @Nonnull
        @Override
        public Builder path(@Nullable String value) {
            return super.path(value);
        }

        @Nonnull
        public Builder propertyKey(@Nullable String value) {
            addIf(NOT_BLANK, propertyKeys, value);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, propertyKeys, value, values);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, propertyKeys, values);

            return self();
        }

        @Override
        protected Builder self() {
            return this;
        }
    }
}
