package com.atlassian.bitbucket.user;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Issued when an admin requests the anonymization of a user's data on this instance.
 * <p>
 * The application's internal user data has been deleted and some data might not be accessible by the user's
 * {@link ApplicationUser#getName() name} anymore.
 * <p>
 * Where possible, the user's {@link ApplicationUser#getId() ID} should be used to find data associated with the user.
 * Alternatively, the {@link #getOriginalUsername() original username} can be used to find data associated with the user.
 * <p>
 * Note that the {@link ApplicationUser} has been deleted, so only the {@link ApplicationUser#getName() username} and
 * {@link ApplicationUser#getId() ID} properties are present and the {@link ApplicationUser#isActive() active flag}
 * is <code>false</code>.
 *
 * @see UserAdminService#eraseUser(String)
 * @since 5.16
 */
public class UserErasureRequest {

    private final ApplicationUser updatedUser;
    private final String originalUsername;

    private UserErasureRequest(Builder builder) {
        updatedUser = builder.newUser;
        originalUsername = builder.previousUsername;
    }

    /**
     * @return the user with the anonymized name
     */
    @Nonnull
    public ApplicationUser getUpdatedUser() {
        return updatedUser;
    }

    /**
     * @return the original username previously associated with the anonymized user
     */
    @Nonnull
    public String getOriginalUsername() {
        return originalUsername;
    }

    public static class Builder {

        private final ApplicationUser newUser;
        private final String previousUsername;

        /**
         * @param updatedUser the user with the anonymized name
         * @param originalUsername the original username previously associated with the anonymized user
         */
        public Builder(@Nonnull ApplicationUser updatedUser, @Nonnull String originalUsername) {
            requireNonNull(updatedUser, "updatedUser");
            requireNonNull(originalUsername, "originalUsername");

            this.newUser = updatedUser;
            this.previousUsername = originalUsername;
        }

        public UserErasureRequest build() {
            return new UserErasureRequest(this);
        }
    }
}
