package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.scm.AbstractMergeCommandParameters;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.util.Map;

/**
 * Parameters for {@link ScmPullRequestCommandFactory#merge(PullRequestMergeCommandParameters) merging} a pull request.
 */
public class PullRequestMergeCommandParameters extends AbstractMergeCommandParameters {

    private final Map<String, Object> context;

    private PullRequestMergeCommandParameters(Builder builder) {
        super(builder);

        context = builder.context.build();
    }

    /**
     * @return additional context provided when the merge was requested
     */
    @Nonnull
    public Map<String, Object> getContext() {
        return context;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final ImmutableMap.Builder<String, Object> context;

        public Builder() {
            context = ImmutableMap.builder();
        }

        @Nonnull
        public PullRequestMergeCommandParameters build() {
            validate();

            return new PullRequestMergeCommandParameters(this);
        }

        /**
         * @param key   the key for the context map entry
         * @param value the value for the context map entry
         * @return {@code this}
         */
        @Nonnull
        public Builder context(@Nonnull String key, @Nonnull String value) {
            context.put(key, value);

            return self();
        }

        /**
         * @param value a map containing additional context, which may not be {@code null} or contain any {@code null}
         *              keys <i>or values</i>
         * @return {@code this}
         */
        @Nonnull
        public Builder context(@Nonnull Map<String, Object> value) {
            context.putAll(value);

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
