package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.BulkCommitsRequest;
import com.atlassian.bitbucket.commit.CommitService;
import com.atlassian.bitbucket.commit.LastModifiedCallback;
import com.atlassian.bitbucket.commit.LastModifiedRequest;
import com.atlassian.bitbucket.compare.CompareRequest;
import com.atlassian.bitbucket.content.ArchiveRequest;
import com.atlassian.bitbucket.content.ContentService;
import com.atlassian.bitbucket.content.PatchRequest;
import com.atlassian.bitbucket.hook.ScmHookHandlerFactory;
import com.atlassian.bitbucket.io.TypeAwareOutputSupplier;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.repository.*;
import com.atlassian.bitbucket.scm.bulk.BulkCommitCallback;
import com.atlassian.bitbucket.scm.bulk.BulkTraversalCallback;
import com.atlassian.bitbucket.scm.bulk.BulkTraverseCommitsCommandParameters;
import com.atlassian.bitbucket.scm.bulk.ScmBulkContentCommandFactory;
import com.atlassian.bitbucket.scm.compare.ScmCompareCommandFactory;
import com.atlassian.bitbucket.scm.mirror.ScmMirrorCommandFactory;
import com.atlassian.bitbucket.scm.pull.ScmPullRequestCommandFactory;
import com.atlassian.bitbucket.scm.ref.ScmRefCommandFactory;

import javax.annotation.Nonnull;
import java.util.Locale;

/**
 * Enumerates optional features an SCM can support.
 */
public enum ScmFeature {

    /**
     * Indicates the SCM supports {@link ScmExtendedCommandFactory#archive streaming archives}.
     *
     * @see ContentService#streamArchive(ArchiveRequest, TypeAwareOutputSupplier)
     * @see ScmExtendedCommandFactory#archive(ArchiveCommandParameters, TypeAwareOutputSupplier)
     * @since 5.1
     */
    ARCHIVE,
    /**
     * Indicates the SCM supports {@link ScmBulkContentCommandFactory#commits bulk commit retrieval}.
     *
     * @see CommitService#streamCommits(BulkCommitsRequest, BulkCommitCallback)
     * @see ScmBulkContentCommandFactory#commits
     * @since 5.8
     */
    BULK_COMMITS,
    /**
     * Indicates the SCM supports {@link ScmBulkContentCommandFactory#contents bulk content retrieval}.
     *
     * @see ScmBulkContentCommandFactory#contents
     * @see ScmService#getBulkContentCommandFactory(Repository)
     * @since 4.2
     */
    BULK_CONTENT,
    /**
     * Indicates the SCM supports {@link ScmBulkContentCommandFactory#traverseCommits bulk commit traversal}.
     *
     * @see ScmBulkContentCommandFactory#traverseCommits(BulkTraverseCommitsCommandParameters, BulkTraversalCallback)
     * @since 5.11
     */
    BULK_TRAVERSAL,
    /**
     * Indicates the SCM supports {@link ScmCommandBuilder builders}.
     *
     * @see ScmService#createBuilder(Repository)
     */
    COMMAND_BUILDERS,
    /**
     * Indicates the SCM provides a {@link ScmCompareCommandFactory command factory}
     * to compare refs.
     * <p>
     * Note: SCMs <i>may</i> support comparing refs {@link #CROSS_REPOSITORY cross-repository}, but they are not
     * <i>required</i> to do so.
     *
     * @see ScmService#getCompareCommandFactory(CompareRequest)
     */
    COMPARE,
    /**
     * Indicates the SCM supports <i>cross-repository</i> operations. This feature applies to:
     * <ul>
     *     <li>{@link #COMPARE}</li>
     *     <li>{@link #MERGE}</li>
     *     <li>{@link #PULL_REQUESTS}</li>
     * </ul>
     * Implementing these features in a cross-repository manner may involve substantial additional complexity,
     * so SCMs are free not to. When this feature is not supported, providing a secondary repository on commands
     * like {@link CommitsCommandParameters commits} or opening a {@link PullRequest pull request} between two
     * repositories will fail.
     *
     * @see #COMPARE
     * @see #FORK
     * @see #MERGE
     * @see #PULL_REQUESTS
     */
    CROSS_REPOSITORY,
    /**
     * Indicates the SCM supports file edit operations.
     * 
     * @see ScmExtendedCommandFactory#editFile(EditFileCommandParameters) 
     * @since 4.13
     */
    EDIT_FILE,
    /**
     * Indicates the SCM supports {@link RepositoryService#fork forking} its repositories.
     * <p>
     * SCMs which do not support forking will not support {@link #CROSS_REPOSITORY cross-repository} operations,
     * but just because forking is supported does <i>not</i> imply cross-repository support. The complexity of
     * cross-repository operations can be substantially higher than the complexity of creating forks, for some
     * SCMs, and forks can still offer value (for example, a sandboxed place to work) even if cross-repository
     * operations are not possible.
     *
     * @see RepositoryService#fork(RepositoryForkRequest)
     * @see #CROSS_REPOSITORY
     */
    FORK,
    /**
     * Indicates the SCM provides a {@link ScmHookHandlerFactory handler factory} to apply repository-level hooks.
     *
     * @see ScmService#getHookHandlerFactory(Repository)
     */
    HOOKS,
    /**
     * Indicates the SCM supports integrity checks
     * @since 4.12
     */
    INTEGRITY_CHECKS,
    /**
     * Indicates the SCM supports streaming the {@link ScmExtendedCommandFactory#lastModified last modification}
     * for files.
     *
     * @see CommitService#streamLastModified(LastModifiedRequest, LastModifiedCallback)
     * @since 4.6
     */
    LAST_MODIFIED,
    /**
     * Indicates the SCM supports {@link ScmExtendedCommandFactory#merge merging branches}.
     * <p>
     * Note: SCMs <i>may</i> support {@link #CROSS_REPOSITORY cross-repository} merges, merging a branch from one
     * repository to a target branch in another, but they are not <i>required</i> to do so.
     *
     * @see ScmExtendedCommandFactory#merge(MergeCommandParameters)
     * @see #CROSS_REPOSITORY
     */
    MERGE,
    /**
     * Indicates the SCM supports selectable merge strategies. Only SCMs which support {@link #MERGE merge commands}
     * and/or {@link #PULL_REQUESTS pull requests} should mark this feature as supported.
     * <p>
     * Generally an SCM should support at least two strategies if this feature is enabled since, with only a single
     * strategy, the strategy isn't actually selectable. But the only <i>requirement</i> of this feature is that the
     * SCM must support <i>at least one</i> strategy.
     *
     * @since 4.9
     */
    MERGE_STRATEGIES,
    /**
     * Indicates the SCM supports {@link ScmMirrorCommandFactory mirroring repositories}
     *
     * @see ScmService#getMirrorCommandFactory(Repository)
     * @since 4.1
     */
    MIRRORS,
    /**
     * Indicates the SCM supports {@link ScmExtendedCommandFactory#patch streaming patches}.
     *
     * @see ContentService#streamPatch(PatchRequest, TypeAwareOutputSupplier)
     * @see ScmExtendedCommandFactory#patch(PatchCommandParameters, TypeAwareOutputSupplier)
     * @since 6.7
     */
    PATCH,
    /**
     * Indicates the SCM provides a {@link ScmPullRequestCommandFactory command factory}
     * to create, view and merge {@link PullRequest pull requests}.
     * <p>
     * Note: SCMs <i>may</i> support {@link #CROSS_REPOSITORY cross-repository} pull requests, merging commits from
     * one repository into another, but they are not <i>required</i> to do so.
     *
     * @see ScmService#getPullRequestCommandFactory(PullRequest)
     * @see #CROSS_REPOSITORY
     */
    PULL_REQUESTS,
    /**
     * Indicates the SCM supports {@link ScmExtendedCommandFactory#push pushing} local repositories to an URL.
     *
     * @see ScmExtendedCommandFactory#push
     * @since 7.11
     */
    PUSH,
    /**
     * Indicates the SCM provides a {@link ScmRefCommandFactory command factory} to create
     * {@link Branch branches} and {@link Tag tags}.
     *
     * @see ScmService#getRefCommandFactory(Repository)
     */
    REFS,
    /**
     * Indicates the SCM supports retrieving
     * {@link ScmExtendedCommandFactory#signedObjects signed objects}.
     *
     * @since 5.1
     */
    SIGNED_OBJECTS,
    /**
     * Indicates the SCM supports {@link ScmExtendedCommandFactory#updateDefaultBranch updating the default branch}
     * in its repositories.
     *
     * @see RefService#setDefaultBranch(Repository, String)
     */
    UPDATE_DEFAULT_BRANCH;

    @Nonnull
    public static ScmFeature parse(@Nonnull String value) {
        //Normalize the value to make this a little more forgiving:
        value = value.trim()             //Trim all whitespace
                .replace("-", "_")       //Replace hyphens with underscores
                .toUpperCase(Locale.US); //Convert to uppercase

        return ScmFeature.valueOf(value);
    }
}
